;;; nnimap.el --- imap backend for Gnus
;; Copyright (C) 1998,1999 Free Software Foundation, Inc.

;; Author: Simon Josefsson <jas@pdc.kth.se>
;;         Jim Radford <radford@robby.caltech.edu>
;; Keywords: mail

;; This file is not part of GNU Emacs.

;; GNU Emacs is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Commentary:

;; Todo (roughly in order of priority):
;;
;;   o What about Gnus's article editing, can we support it?
;;   o Verify that we don't use IMAP4rev1 specific things (RFC2060 App B)
;;   o Dont uid fetch 1,* in nnimap-retrive-groups (slow)
;;   o Split up big fetches (1,* header especially) in smaller chunks
;;   o Use \Draft to support the draft group??
;;   o What do I do with gnus-newsgroup-*?
;;   o Tell Gnus about new groups (how can we tell?)
;;   o Add asynchronous support
;;   o Bulletin board stuff.
;;   o Respooling (fix Gnus?) (unnecessery?)
;;   o Add support for the following: (if applicable)
;;       request-list-newsgroups, request-regenerate
;;       list-active-group,
;;       request-associate-buffer, request-restore-buffer,
;;   o Do The Right Thing when UIDVALIDITY changes
;;   o Split to other backends, different split rules for different
;;     servers/inboxes
;;   o Fix Gnus to handle leading '.' in group names
;;   o Support RFC2221 (Login referrals)
;;   o IMAP2BIS compatibility? (RFC2061)
;;   o ACAP stuff (perhaps a different project, would be nice to ACAPify 
;;     .newsrc.eld)
;;   o MIME

(eval-and-compile
  (require 'imap))

(require 'nnoo)
(require 'nnmail)
(require 'gnus)
(require 'gnus-range)

(nnoo-declare nnimap)

(gnus-declare-backend "nnimap" 'mail 'address 'prompt-address
		      'physical-address)

(defconst nnimap-version "nnimap 0.120")

(defvoo nnimap-address nil
  "Address of physical IMAP server.  If nil, use the virtual server's name.")

(defvoo nnimap-server-port nil
  "Port number on physical IMAP server.  If nil, defaults to 993 for
SSL connections and 143 otherwise.")

;; Splitting variables

(defvar nnimap-split-crosspost t
  "If non-nil, do crossposting if several split methods match the mail.
If nil, the first match found will be used.")

(defvar nnimap-split-inbox nil
  "*Name of mailbox to split mail from. Mail is read from this mailbox and
split according to rules in nnimap-split-rules.

This can be a string or a list of strings.")

(defvar nnimap-split-rule nil
  "*Mail will be split according to theese rules. Mail is read from mailbox(es)
specified in nnimap-split-inbox.

If you'd like, for instance, one mail group for mail from the
\"gnus-imap\" mailing list, one group for junk mail and leave
everything else in the incoming mailbox, you could do something like
this:

(setq nnimap-split-rule '((\"INBOX.gnus-imap\"   \"From:.*gnus-imap\")
			  (\"INBOX.junk\"        \"Subject:.*buy\")))

As you can see, nnimap-split-rule is a list of lists, where the first
element in each \"rule\" is the name of the IMAP mailbox, and the
second is a regexp that nnimap will try to match on the header to find
a fit.

The first element can also be a list. In that case, the first element
is the server the second element is the group on that server in which
the matching article will be stored.

The second element can also be a function.  In that case, it will be
called narrowed to the headers with the first element of the rule as
the argument.  It should return a non-nil value if it thinks that the
mail belongs in that group.")

;; Authorization / Privacy variables

(defvoo nnimap-auth-method nil
  "Obsolete.")

(defvoo nnimap-stream nil
  "How nnimap will connect to the server.

The default, nil, will try to use the \"best\" method the server can
handle.

Change this if

1) you want to connect with SSL. The SSL integration with IMAP is
   brain-dead so you'll have to tell it specifically.

2) your server is more capable than your environment -- i.e. your
   server accept Kerberos login's but you haven't installed the
   `imtest' program or your machine isn't configured for Kerberos.

Possible choices: kerberos4, ssl, network")

(defvoo nnimap-authenticator nil
  "How nnimap authenticate itself to the server.

The default, nil, will try to use the \"best\" method the server can
handle.

There is only one reason for fiddling with this variable, and that is
if your server is more capable than your environment -- i.e. you
connect to a server that accept Kerberos login's but you haven't
installed the `imtest' program or your machine isn't configured for
Kerberos.

Possible choices: kerberos4, cram-md5, login, anonymous.")

(defvoo nnimap-directory gnus-directory
  "Directory to keep NOV cache files for nnimap groups.")

(defvoo nnimap-nov-file-name "overview.nnimap."
  "NOV cache base filename. The group name will be appended. A typical
complete file name would be ~/News/overview.nnimap.pdc.INBOX.ding, or
~/News/overview/nnimap/pdc/INBOX/ding if `gnus-use-long-file-name' is
nil")

(defvoo nnimap-nov-is-evil t
  "Disable usage of NOV caches")

(defvoo nnimap-expunge-on-close 'always ; 'ask, 'never
  "When a IMAP group with articles marked for deletion is closed, this
variable determine if nnimap should actually remove the articles or
not.

If always, nnimap always perform a expunge when closing the group.
If never, nnimap never expunges articles marked for deletion.
If ask, nnimap will ask you if you wish to expunge marked articles.

When setting this variable to `never', you can only expunge articles
by using `G x' (gnus-group-nnimap-expunge) from the Group buffer.")

(defvoo nnimap-list-pattern "*" 
  "A string LIMIT or list of strings with mailbox wildcards used to
limit available groups. Se below for available wildcards.

The LIMIT string can be a cons cell (REFERENCE . LIMIT), where
REFERENCE will be passed as the first parameter to LIST/LSUB. The
semantics of this are server specific, on the University of Washington
server you can specify a directory.

Example:
 '(\"INBOX\" \"mail/*\" (\"~friend/mail/\" . \"list/*\"))

There are two wildcards * and %. * matches everything, % matches
everything in the current hierarchy.")

(defvoo nnimap-server-address nil
  "Obsolete. Use `nnimap-address'.")

(defcustom nnimap-authinfo-file "~/.authinfo"
  "Authorization information for IMAP servers. In .netrc format."
  :type
  '(choice file
	   (repeat :tag "Entries"
		   :menu-tag "Inline"
		   (list :format "%v"
			 :value ("" ("login" . "") ("password" . ""))
			 (string :tag "Host")
			 (checklist :inline t
				    (cons :format "%v"
					  (const :format "" "login")
					  (string :format "Login: %v"))
				    (cons :format "%v"
					  (const :format "" "password")
					  (string :format "Password: %v")))))))

;; Internal variables:

(defvar nnimap-debug "*nnimap-debug*")
(defvar nnimap-current-move-server nil)
(defvar nnimap-current-move-group nil)
(defvar nnimap-current-move-article nil)
(defvar nnimap-length)
(defvar nnimap-counter)
(defvar nnimap-uids)

;; Various server variables.


;; Internal variables.
(defvar nnimap-server-buffer-alist nil) ;; Map server name to buffers.
(defvar nnimap-current-server nil)      ;; Current server
(defvar nnimap-server-buffer nil)       ;; Current servers' buffer

(nnoo-define-basics nnimap)

;; Utility functions:

(defsubst nnimap-get-server-buffer (server)
  "Return buffer for SERVER. If SERVER is nil, the current server is
used."
  (cadr (assoc (or server nnimap-current-server) nnimap-server-buffer-alist)))

(defun nnimap-possibly-change-server (server)
  ;; Return buffer of server SERVER. If SERVER is nil, return current
  ;; server buffer. Changes the current server as a side-effect.
  (setq nnimap-current-server (or server nnimap-current-server)
	nnimap-server-buffer (nnimap-get-server-buffer nnimap-current-server)))

(defun nnimap-verify-uidvalidity (group server)
  (let* ((gnusgroup (gnus-group-prefixed-name 
		     group (gnus-server-to-method 
			    (format "nnimap:%s" server))))
	 (new-uidvalidity (imap-mailbox-get 'uidvalidity))
	 (old-uidvalidity (gnus-group-get-parameter gnusgroup 'uidvalidity)))
    (if old-uidvalidity
	(if (not (equal old-uidvalidity new-uidvalidity))
	    nil ;; uidvalidity clash
	  (gnus-group-set-parameter gnusgroup 'uidvalidity new-uidvalidity)
	  t)
      (gnus-group-add-parameter gnusgroup (cons 'uidvalidity new-uidvalidity))
      t)))

(defun nnimap-find-minmax-uid (group &optional examine)
  (with-current-buffer nnimap-server-buffer
    (when (imap-mailbox-select group nil examine)
      (let (minuid maxuid)
	(when (> (imap-mailbox-get 'exists) 0)
	  (imap-fetch "1,*" "UID" nil 'nouidfetch)
	  (imap-message-map (lambda (uid Uid)
			      (setq minuid (if minuid (min minuid uid) uid)
				    maxuid (if maxuid (max maxuid uid) uid)))
			    'UID))
	(list (imap-mailbox-get 'exists) minuid maxuid)))))
  
(defun nnimap-possibly-change-group (group &optional server)
  (when (nnimap-possibly-change-server server)
    (with-current-buffer nnimap-server-buffer
      (if (null group)
	  imap-current-mailbox
	(if (imap-current-mailbox-p group)
	    t
	  (when imap-current-mailbox
	    (nnimap-expunge-close-group))
	  (when (imap-mailbox-select group)
	    (if (nnimap-verify-uidvalidity group 
					   (or server nnimap-current-server))
		imap-current-mailbox
	      (imap-mailbox-unselect)
	      (gnus-message 1 "nnimap: Group %s is not uid-valid." group)
	      (ding)
	      nil)))))))

(defun nnimap-replace-whitespace (string)
  "Return STRING with all whitespace replaced with space."
  (when string
    (while (string-match "[\r\n\t]+" string)
      (setq string (replace-match " " t t string)))
    string))

;; Required backend functions

(defun nnimap-retrieve-headers-progress ()
  (and (numberp nnmail-large-newsgroup)
       (> nnimap-length nnmail-large-newsgroup)
       (zerop (% (incf nnimap-counter) 20))
       (nnheader-message 6 "nnimap: Receiving headers... %d%%"
			 (/ (* nnimap-counter 100) nnimap-length)))
  (with-current-buffer nntp-server-buffer
    (nnheader-insert-nov
     (with-current-buffer nnimap-server-buffer
       (vector imap-current-message
	       (nnimap-replace-whitespace
		(imap-message-envelope-subject imap-current-message))
	       (nnimap-replace-whitespace
		(imap-envelope-from
		 (car-safe (imap-message-envelope-from
			    imap-current-message))))
	       (nnimap-replace-whitespace 
		(imap-message-envelope-date imap-current-message))
	       (nnimap-replace-whitespace
		(imap-message-envelope-message-id imap-current-message))
	       (nnimap-replace-whitespace
		(let ((str (if (imap-capability 'IMAP4rev1)
			       (nth 2 (assoc
				       "HEADER.FIELDS REFERENCES"
				       (imap-message-get
					imap-current-message 'BODYDETAIL)))
			     (imap-message-get imap-current-message 
					       'RFC822.HEADER))))
		  (if (> (length str) (length "References: "))
		      (substring str (length "References: "))
		    (if (and (setq str (imap-message-envelope-in-reply-to
					imap-current-message))
			     (string-match "<[^>]+>" str))
			(substring str (match-beginning 0) (match-end 0))))))
	       (imap-message-get imap-current-message 'RFC822.SIZE)
	       (imap-body-lines (imap-message-body imap-current-message))
	       nil ;; xref
	       nil))))) ;; extra-headers

(defun nnimap-retrieve-headers-store-uids ()
  (setq nnimap-uids (cons imap-current-message nnimap-uids)))

(defun nnimap-retrieve-headers-get-uids (articles fetch-old)
  (with-current-buffer nnimap-server-buffer
    (if (numberp (car articles))
	(let ((imap-fetch-data-hook '(nnimap-retrieve-headers-store-uids))
	      nnimap-uids)
	  (if (and fetch-old (not (numberp fetch-old)))
	      (imap-fetch "1:*" "UID")
	    (imap-fetch (nnimap-range-to-string
			 (gnus-compress-sequence articles t)) "UID"))
	  (nreverse nnimap-uids))
      (mapcar (lambda (msgid)
		(imap-search 
		 (format "HEADER Message-Id %s" msgid)))
	      articles))))

(defun nnimap-group-overview-filename (group server)
  "Make pathname for GROUP."
  (let ((dir (file-name-as-directory (expand-file-name nnimap-directory)))
	(file (concat nnimap-nov-file-name server "." 
		      (nnheader-translate-file-chars group))))
    (if (or nnmail-use-long-file-names
	    (file-directory-p (concat dir file)))
	(concat dir file)
      (concat dir (mm-encode-coding-string
		   (nnheader-replace-chars-in-string file ?. ?/)
		   nnmail-pathname-coding-system)))))

(defun nnimap-retrieve-headers-from-file (articles group server)
  (with-current-buffer nntp-server-buffer
    (erase-buffer)
    (let ((nov (nnimap-group-overview-filename group server)))
      (when (file-exists-p nov)
	(nnheader-insert-file-contents nov)
	(nnheader-nov-delete-outside-range (car articles) 
					   (car (last articles)))))))

(deffoo nnimap-retrieve-headers (articles &optional group server fetch-old)
  (when (nnimap-possibly-change-group group server)
    (with-current-buffer nntp-server-buffer
      (erase-buffer))
    (let ((uids (nnimap-retrieve-headers-get-uids articles fetch-old))
	  (imap-fetch-data-hook '(nnimap-retrieve-headers-progress))
	  (nnimap-length (length articles))
	  (nnimap-counter 0))
      ;;(nnimap-retrieve-headers-from-file uids group server)
      ;; xxx only imap-fetch articles we don't have
      (imap-fetch (nnimap-range-to-string (gnus-compress-sequence uids t))
		  (concat 
		   "(UID RFC822.SIZE ENVELOPE BODY "
		   (if (imap-capability 'IMAP4rev1 nnimap-server-buffer)
		       "BODY.PEEK[HEADER.FIELDS (References)])"
		     "RFC822.HEADER.LINES (References))"))
		  nil nil nnimap-server-buffer)
      ;; xxx add new nov lines to cache
      (and (numberp nnmail-large-newsgroup)
	   (> nnimap-length nnmail-large-newsgroup)
	   (nnheader-message 6 "nnimap: Receiving headers...done"))
      'nov)))

(defun nnimap-open-connection (server)
  (if (not (imap-open nnimap-address nnimap-server-port nnimap-stream
		      nnimap-authenticator nnimap-server-buffer))
      (nnheader-report 'nnimap "Can't open connection to server %s" server)
    (unless (or (imap-capability 'IMAP4 nnimap-server-buffer)
		(imap-capability 'IMAP4rev1 nnimap-server-buffer))
      (imap-close nnimap-server-buffer)
      (nnheader-report 'nnimap "Server %s is not IMAP4 compliant" server))
    (let (list alist user passwd)
      (and (fboundp 'gnus-parse-netrc)
	   (setq list (gnus-parse-netrc nnimap-authinfo-file)
		 alist (or (and (gnus-netrc-get 
				 (gnus-netrc-machine list server) "machine")
				(gnus-netrc-machine list server))
			   (gnus-netrc-machine list nnimap-address))
		 user (gnus-netrc-get alist "login")
		 passwd (gnus-netrc-get alist "password")))
      (if (imap-authenticate nnimap-server-buffer user passwd)
	  (prog1
	      (push (list server nnimap-server-buffer)
		    nnimap-server-buffer-alist)
	    (nnimap-possibly-change-server server))
	(imap-close nnimap-server-buffer)
	(kill-buffer nnimap-server-buffer)
	(nnheader-report 'nnimap "Could not authenticate to %s" server)))))

(deffoo nnimap-open-server (server &optional defs)
  (nnheader-init-server-buffer)
  (if (nnimap-server-opened server)
      t
    (unless (assq 'nnimap-server-buffer defs)
      (push (list 'nnimap-server-buffer (concat " *nnimap* " server)) defs))
    ;; translate `nnimap-server-address' to `nnimap-address' in defs
    ;; for people that configured nnimap with a very old version
    (unless (assq 'nnimap-address defs)
      (if (assq 'nnimap-server-address defs)
	  (push (list 'nnimap-address
		      (cadr (assq 'nnimap-server-address defs))) defs)
	(push (list 'nnimap-address server) defs)))
    (nnoo-change-server 'nnimap server defs)
    (if (null nnimap-server-buffer)
	(error "this shouldn't happen"))
    (or (imap-opened nnimap-server-buffer)
	(nnimap-open-connection server))))

(deffoo nnimap-server-opened (&optional server)
  "If SERVER is the current virtual server, and the connection to the
physical server is alive, this function return a non-nil value. If
SERVER is nil, it is treated as the current server."
  ;; clean up autologouts??
  (and (or server nnimap-current-server)
       (nnoo-server-opened 'nnimap (or server nnimap-current-server))
       (imap-opened (nnimap-get-server-buffer server))))

(deffoo nnimap-close-server (&optional server)
  "Close connection to server and free all resources connected to
it. Return nil if the server couldn't be closed for some reason."
  (let ((server (or server nnimap-current-server)))
    (when (or (nnimap-server-opened server)
	      (imap-opened (nnimap-get-server-buffer server)))
      (imap-close (nnimap-get-server-buffer server))
      (kill-buffer (nnimap-get-server-buffer server))
      (setq nnimap-server-buffer nil
	    nnimap-current-server nil
	    nnimap-server-buffer-alist 
	    (delq server nnimap-server-buffer-alist)))
    (nnoo-close-server 'nnimap server)))

(deffoo nnimap-request-close ()
  "Close connection to all servers and free all resources that the
backend have reserved. All buffers that have been created by that
backend should be killed. (Not the nntp-server-buffer, though.) This
function is generally only called when Gnus is shutting down."
  (mapcar (lambda (server) (nnimap-close-server (car server)))
	  nnimap-server-buffer-alist)
  (setq nnimap-server-buffer-alist nil))

(deffoo nnimap-status-message (&optional server)
  "This function returns the last error message from server."
  (when (nnimap-possibly-change-server server)
    (nnoo-status-message 'nnimap server)))

(defun nnimap-demule (string)
  (funcall (if (and (fboundp 'string-as-multibyte)
		    (subrp (symbol-function 'string-as-multibyte)))
	       'string-as-multibyte
	     'identity)
	   string))

(defun nnimap-request-article-part (article part prop
					    &optional group server to-buffer)
  (when (nnimap-possibly-change-group group server)
    (with-current-buffer (or to-buffer nntp-server-buffer)
      (erase-buffer)
      (and (stringp article)
	   (setq article (car-safe (imap-search
				    (format "HEADER Message-Id %s" article)
				    nnimap-server-buffer))))
      (when article
	(gnus-message 9 "nnimap: Fetching (part of) article %d..." article)
	(insert (nnimap-demule 
		 (or (imap-fetch article part prop nil nnimap-server-buffer)
		     "")))
	(nnheader-ms-strip-cr)
	(gnus-message 9 "nnimap: Fetching (part of) article %d...done"
		      article))
      (if (bobp)
	  (nnheader-report 'nnimap "No such article")
	(cons group article)))))

(deffoo nnimap-request-article (article &optional group server to-buffer)
  (nnimap-request-article-part 
   article "RFC822.PEEK" 'RFC822 group server to-buffer))

(deffoo nnimap-request-head (article &optional group server to-buffer)
  (nnimap-request-article-part 
   article "RFC822.HEADER" 'RFC822.HEADER group server to-buffer))

(deffoo nnimap-request-body (article &optional group server to-buffer)
  (nnimap-request-article-part
   article "RFC822.TEXT.PEEK" 'RFC822.TEXT group server to-buffer))

(deffoo nnimap-request-group (group &optional server fast)
  (nnimap-request-update-info-internal
   group
   (gnus-get-info (gnus-group-prefixed-name 
		   group (gnus-server-to-method (format "nnimap:%s" server))))
   server)
  (when (nnimap-possibly-change-group group server)
    (with-current-buffer nnimap-server-buffer
      (unless fast
	(let (info)
	  (when (setq info (nnimap-find-minmax-uid group))
	    (nnheader-insert "211 %d %d %d %s\n" (or (nth 0 info) 0)
			     (max 1 (or (nth 1 info) 1))
			     (or (nth 2 info) 0) group))))
      (nnheader-report 'nnimap "Group %s selected" group)
      t)))
      
(defun nnimap-expunge-close-group (&optional server)
  (with-current-buffer nnimap-server-buffer
    (when (nnimap-possibly-change-server server)
      (case nnimap-expunge-on-close
	('always (imap-mailbox-expunge)
		 (imap-mailbox-close))
	('ask (if (and (imap-search "DELETED")
		       (gnus-y-or-n-p (format 
				       "Expunge articles in group `%s'? "
				       imap-current-mailbox)))
		  (progn (imap-mailbox-expunge)
			 (imap-mailbox-close))
		(imap-mailbox-unselect)))
	(t (imap-mailbox-unselect)))
      (not imap-current-mailbox))))

(deffoo nnimap-close-group (group &optional server)
  (nnimap-expunge-close-group server))

(defun nnimap-pattern-to-list-arguments (pattern)
  (mapcar (lambda (p)
	    (cons (car-safe p) (or (cdr-safe p) p)))
	  (if (and (listp pattern)
		   (listp (cdr pattern)))
	      pattern
	    (list pattern))))

(deffoo nnimap-request-list (&optional server)
  (when (nnimap-possibly-change-server server)
    (with-current-buffer nntp-server-buffer
      (gnus-message 5 "nnimap: Generating active list for %s..." server)
      (erase-buffer)
      (dolist (pattern (nnimap-pattern-to-list-arguments 
			nnimap-list-pattern))
	(dolist (mbx (imap-mailbox-list 
		      nnimap-server-buffer (cdr pattern) t (car pattern)))
	  (or (member "\\NoSelect" 
		      (imap-mailbox-get 'list-flags mbx nnimap-server-buffer))
	      ;; Quote mailboxes if they contain SPC
	      ;; xxx relies on internal knowledge of gnus, breaks nntp rfc
	      ;; We lie about high/low article number
	      (if (string-match " " mbx)
		  (insert (format "\"%s\" 0 1 y\n" mbx))
		(insert (format "%s 0 1 y\n" mbx))))))
      (gnus-message 5 "nnimap: Generating active list for %s...done" server))
    t))

(deffoo nnimap-request-post (&optional server)
  (nnheader-report 'nnimap "Nnimap doesn't support posting"))

;; Optional backend functions

(deffoo nnimap-retrieve-groups (groups &optional server)
  (when (nnimap-possibly-change-server server)
    (gnus-message 5 "nnimap: Checking mailboxes...")
    (with-current-buffer nntp-server-buffer
      (erase-buffer)
      (dolist (group groups)
	(gnus-message 7 "nnimap: Checking mailbox %s" group)
	(or (member "\\NoSelect" 
		    (imap-mailbox-get 'list-flags group nnimap-server-buffer))
	    (let ((info (nnimap-find-minmax-uid group 'examine)))
	      (insert (format "211 %d %d %d %s\n" (or (nth 0 info) 0)
			      (max 1 (or (nth 1 info) 1))
			      (or (nth 2 info) 0) group))))))
    (gnus-message 5 "nnimap: Checking mailboxes...done")
    'groups))

(deffoo nnimap-request-update-info-internal (group info &optional server)
  (when (nnimap-possibly-change-group group server)
    (when info ;; xxx what does this mean? should we create a info?
      (with-current-buffer nnimap-server-buffer
	(gnus-message 5 "nnimap: Updating info for %s..."
		      (gnus-info-group info))
	
	(when (nnimap-mark-permanent-p 'read)
	  (gnus-info-set-read 
	   info
	   (let (seen unseen)
	     ;; read info could contain articles marked unread by other
	     ;; imap clients!  we correct this
	     (setq seen (gnus-uncompress-range (gnus-info-read info))
		   unseen (imap-search "UNSEEN UNDELETED")
		   seen (gnus-set-difference seen unseen)
	     ;; seen might lack articles marked as read by other
	     ;; imap clients! we correct this
		   seen (append seen (imap-search "SEEN"))
	     ;; remove dupes
		   seen (sort seen '<)
		   seen (gnus-compress-sequence seen t))
	     ;; we can't return '(1) since this isn't a "list of ranges",
	     ;; and we can't return '((1)) since gnus-list-of-unread-articles
	     ;; is buggy so we return '((1 . 1)).
	     (if (and (integerp (car seen))
		      (null (cdr seen)))
		 (list (cons (car seen) (car seen)))
	       seen))))

	(mapc (lambda (pred)
		(when (and (nnimap-mark-permanent-p (cdr pred))
			   (member (nnimap-mark-to-flag (cdr pred))
				   (imap-mailbox-get 'flags)))
		  (gnus-info-set-marks
		   info
		   (nnimap-update-alist-soft
		    (cdr pred)
		    (gnus-compress-sequence
		     (imap-search (nnimap-mark-to-predicate (cdr pred))))
		    (gnus-info-marks info))
		   t)))
	      gnus-article-mark-lists)
	
	(gnus-message 5 "nnimap: Updating info for %s...done"
		      (gnus-info-group info))
	
	info))))

(deffoo nnimap-request-type (group &optional article)
  'mail)

(deffoo nnimap-request-set-mark (group actions &optional server)
  (when (nnimap-possibly-change-group group server)
    (with-current-buffer nnimap-server-buffer
      (let (action)
	(gnus-message 7 "nnimap: Setting marks in %s:%s..."  
		      (nnoo-current-server 'nnimap) group)
	(while (setq action (pop actions))
	  (let ((range (nth 0 action))
		(what  (nth 1 action))
		(cmdmarks (nth 2 action))
		marks)
	    ;; cache flags are pointless on the server
	    (setq cmdmarks (delq 'cache cmdmarks))
	    ;; flag dormant articles as ticked
	    (if (memq 'dormant cmdmarks)
		(setq cmdmarks (cons 'tick cmdmarks)))
	    ;; remove stuff we are forbidden to store
	    (mapcar (lambda (mark)
		      (if (imap-message-flag-permanent-p
			   (nnimap-mark-to-flag mark))
			  (setq marks (cons mark marks))))
		    cmdmarks)
	    (when (and range marks)
	      (cond ((eq what 'del)
		     (imap-message-flags-del
		      (nnimap-range-to-string range)
		      (nnimap-mark-to-flag marks nil t)))
		    ((eq what 'add)
		     (imap-message-flags-add 
		      (nnimap-range-to-string range)
		      (nnimap-mark-to-flag marks nil t)))
		    ((eq what 'set)
		     (imap-message-flags-set
		      (nnimap-range-to-string range)
		      (nnimap-mark-to-flag marks nil t)))))))
	(gnus-message 7 "nnimap: Setting marks in %s:%s...done" 
		      (nnoo-current-server 'nnimap) group))))
  nil)

(defun nnimap-split-to-groups (rules)
  ;; tries to match all rules in nnimap-split-rule against content of 
  ;; nntp-server-buffer, returns a list of groups that matched.
  (with-current-buffer nntp-server-buffer
    ;; Fold continuation lines.
    (goto-char (point-min))
    (while (re-search-forward "\\(\r?\n[ \t]+\\)+" nil t)
      (replace-match " " t t))
    (if (functionp rules)
	(funcall rules)
      (let (to-groups regrepp)
	(catch 'split-done
	  (dolist (rule rules to-groups)
	    (let ((group (car rule))
		  (regexp (cadr rule)))
	      (goto-char (point-min))
	      (when (and (if (stringp regexp)
			     (progn
			       (setq regrepp (string-match "\\\\[0-9&]" group))
			       (re-search-forward regexp nil t))
			   (funcall regexp group))
			 ;; Don't enter the article into the same group twice.
			 (not (assoc group to-groups)))
		(push (if regrepp
			  (nnmail-expand-newtext group)
			group)
		      to-groups)
		(or nnimap-split-crosspost
		    (throw 'split-done to-groups))))))))))
  
(defun nnimap-split-find-rule (server inbox)
  nnimap-split-rule)

(defun nnimap-split-find-inbox (server)
  (if (listp nnimap-split-inbox)
      nnimap-split-inbox
    (list nnimap-split-inbox)))

(defun nnimap-split-articles (&optional group server)
  (when (nnimap-possibly-change-server server)
    (with-current-buffer nnimap-server-buffer
      (let (rule inbox removeorig (inboxes (nnimap-split-find-inbox server)))
	;; iterate over inboxes
	(while (and (setq inbox (pop inboxes))
		    (nnimap-possibly-change-group inbox)) ;; SELECT
	  ;; find split rule for this server / inbox
	  (when (setq rule (nnimap-split-find-rule server inbox))
	    ;; iterate over articles
	    (dolist (article (imap-search "UNSEEN UNDELETED"))
	      (when (nnimap-request-head article)
		;; copy article to right group(s)
		(setq removeorig nil)
		(dolist (to-group (nnimap-split-to-groups rule))
		  (if (imap-message-copy (number-to-string article)
					 to-group nil nil t)
		      (progn
			(message "IMAP split moved %s:%s:%d to %s" server inbox
				 article to-group)
			(setq removeorig t))
		    (message "IMAP split failed to move %s:%s:%d to %s" server
			     inbox article to-group)))
		;; remove article if it was successfully copied somewhere
		(and removeorig
		     (imap-message-flags-add (format "%d" article)
					     "\\Seen \\Deleted")))))
	  (when (imap-mailbox-select inbox) ;; just in case
	    ;; todo: UID EXPUNGE (if available) to remove splitted articles
	    (nnimap-expunge-close-group)))
	t))))

(deffoo nnimap-request-scan (&optional group server)
  (nnimap-split-articles group server))

(deffoo nnimap-request-create-group (group &optional server args)
  (when (nnimap-possibly-change-server server)
    (or (imap-mailbox-status group 'uidvalidity nnimap-server-buffer)
	(imap-mailbox-create group nnimap-server-buffer))))

(defun nnimap-time-substract (time1 time2)
  "Return TIME for TIME1 - TIME2."
  (let* ((ms (- (car time1) (car time2)))
	 (ls (- (nth 1 time1) (nth 1 time2))))
    (if (< ls 0)
	(list (- ms 1) (+ (expt 2 16) ls))
      (list ms ls))))

(defun nnimap-date-days-ago (daysago)
  "Return date, in format \"3-Aug-1998\", for DAYSAGO days ago."
  (let ((date (format-time-string "%d-%b-%Y" 
				  (nnimap-time-substract
				   (current-time)
				   (days-to-time daysago)))))
    (if (eq ?0 (string-to-char date))
	(substring date 1)
      date)))

(defun nnimap-request-expire-articles-progress ()
  (gnus-message 5 "nnimap: Marking article %d for deletion..." 
		imap-current-message))

;; Notice that we don't actually delete anything, we just mark them deleted.
(deffoo nnimap-request-expire-articles (articles group &optional server force)
  (let ((artseq (gnus-compress-sequence articles)))
    (when (and artseq (nnimap-possibly-change-group group server))
      (with-current-buffer nnimap-server-buffer
	(if force
	    (and (imap-message-flags-add 
		  (nnimap-range-to-string artseq) "\\Deleted")
		 (setq articles nil))
	  (let ((days (or (and nnmail-expiry-wait-function
			       (funcall nnmail-expiry-wait-function group))
			  nnmail-expiry-wait)))
	    (cond ((eq days 'immediate)
		   (and (imap-message-flags-add 
			 (nnimap-range-to-string artseq) "\\Deleted")
			(setq articles nil)))
		  ((numberp days)
		   (let ((oldarts (imap-search
				   (format "UID %s NOT SINCE %s"
					   (nnimap-range-to-string artseq)
					   (nnimap-date-days-ago days))))
			 (imap-fetch-data-hook 
			  '(nnimap-request-expire-articles-progress)))
		     (and oldarts
			  (imap-message-flags-add
			   (nnimap-range-to-string 
			    (gnus-compress-sequence oldarts))
			   "\\Deleted")
			  (setq articles (gnus-set-difference
					  articles oldarts)))))))))))
  ;; return articles not deleted
  articles)

(deffoo nnimap-request-move-article (article group server
					     accept-form &optional last)
  (when (nnimap-possibly-change-server server)
    (save-excursion
      (let ((buf (get-buffer-create " *nnimap move*"))
	    (nnimap-current-move-article article)
	    (nnimap-current-move-group group)
	    (nnimap-current-move-server nnimap-current-server)
	    result)
	(and (nnimap-request-article article group server)
	     (save-excursion
	       (set-buffer buf)
	       (buffer-disable-undo (current-buffer))
	       (insert-buffer-substring nntp-server-buffer)
	       (setq result (eval accept-form))
	       (kill-buffer buf)
	       result)
	     (nnimap-request-expire-articles (list article) group server t))
	result))))
  
(deffoo nnimap-request-accept-article (group &optional server last)
  (when (nnimap-possibly-change-server server)
    (let (uid)
      (and (setq uid
		 (if (string= nnimap-current-server nnimap-current-move-server)
		     ;; moving article within same server, speed it up...
		     (and (nnimap-possibly-change-group
			   nnimap-current-move-group)
			  (imap-message-copy (number-to-string
					      nnimap-current-move-article)
					     group nnimap-server-buffer))
		   ;; turn into rfc822 format (\r\n eol's)
		   (with-current-buffer (current-buffer)
		     (goto-char (point-min))
		     (while (search-forward "\n" nil t)
		       (replace-match "\r\n")))
		   ;; next line for Cyrus server bug
		   (imap-mailbox-unselect nnimap-server-buffer)
		   (imap-message-append group (current-buffer)
					nnimap-server-buffer)))
	   (cons group (nth 1 uid))))))

(deffoo nnimap-request-delete-group (group force &optional server)
  (when (nnimap-possibly-change-server server)
    (if force
	(imap-mailbox-delete group nnimap-server-buffer)
      ;; UNSUBSCRIBE?
      t)))

(deffoo nnimap-request-rename-group (group new-name &optional server)
  (when (nnimap-possibly-change-server server)
    (imap-mailbox-rename group new-name nnimap-server-buffer)))

;;; Gnus functions

(defun gnus-group-nnimap-expunge (group)
  "Expunge deleted articles in current nnimap GROUP."
  (interactive (list (gnus-group-group-name)))
  (let ((mailbox (gnus-group-real-name group))
	method)
    (unless group
      (error "No group on current line"))
    (unless (gnus-get-info group)
      (error "Killed group; can't be edited"))
    (unless (eq 'nnimap (car (setq method (gnus-find-method-for-group group))))
      (error "Expunging only available for nnimap groups"))
    (when (nnimap-possibly-change-group mailbox (cadr method))
      (imap-mailbox-expunge nnimap-server-buffer))))

(defun gnus-group-nnimap-edit-acl (group)
  "Edit the Access Control List of current nnimap GROUP."
  (interactive (list (gnus-group-group-name)))
  (let ((mailbox (gnus-group-real-name group))
	method acl)
    (unless group
      (error "No group on current line"))
    (unless (gnus-get-info group)
      (error "Killed group; can't be edited"))
    (unless (eq 'nnimap (car (setq method (gnus-find-method-for-group group))))
      (error "ACL editing only available for nnimap groups"))
    (when (nnimap-possibly-change-server (cadr method))
      (unless (imap-capability 'ACL nnimap-server-buffer)
	(error "Your server does not support ACL editing"))
      (gnus-edit-form (setq acl (imap-mailbox-acl-get mailbox 
						      nnimap-server-buffer))
		      (format "Editing the access control list for `%s'.

   An access control list is a list of (identifier . rights) elements.

   The identifier string specifies the corresponding user. The
   identifier \"anyone\" is reserved to refer to the universal identity.

   Rights is a string listing a (possibly empty) set of alphanumeric
   characters, each character listing a set of operations which is being
   controlled. Letters are reserved for ``standard'' rights, listed
   below.  Digits are reserved for implementation or site defined rights.

   l - lookup (mailbox is visible to LIST/LSUB commands)
   r - read (SELECT the mailbox, perform CHECK, FETCH, PARTIAL,
       SEARCH, COPY from mailbox)
   s - keep seen/unseen information across sessions (STORE SEEN flag)
   w - write (STORE flags other than SEEN and DELETED)
   i - insert (perform APPEND, COPY into mailbox)
   p - post (send mail to submission address for mailbox,
       not enforced by IMAP4 itself)
   c - create (CREATE new sub-mailboxes in any implementation-defined
       hierarchy)
   d - delete (STORE DELETED flag, perform EXPUNGE)
   a - administer (perform SETACL)" group)
		      `(lambda (form) 
			 (gnus-group-nnimap-edit-acl-done 
			  ,mailbox ',method ',acl form))))))

(defun gnus-group-nnimap-edit-acl-done (mailbox method old-acls new-acls)
  (when (nnimap-possibly-change-server (cadr method))
    (with-current-buffer nnimap-server-buffer
      ;; delete all removed identifiers
      (mapcar (lambda (old-acl)
		(unless (assoc (car old-acl) new-acls)
		    (or (imap-mailbox-acl-delete (car old-acl) mailbox)
			(error "Can't delete ACL for %s..." (car old-acl)))))
	      old-acls)
      ;; set all changed acl's
      (mapcar (lambda (new-acl)
		(let ((new-rights (cdr new-acl))
		      (old-rights (cdr (assoc (car new-acl) old-acls))))
		(unless (and old-rights new-rights
			     (string= old-rights new-rights))
		  (or (imap-mailbox-acl-set (car new-acl) new-rights mailbox)
		      (error "Can't set ACL for %s to %s..." (car new-acl)
			     new-rights)))))
	      new-acls)
      t)))

;;; Gnus glue

(defun nnimap-group-mode-hook ()
  (define-key gnus-group-mode-map (if (fboundp 'kbd) (kbd "G l")
				    (read-kbd-macro "G l"))
    'gnus-group-nnimap-edit-acl)
  (define-key gnus-group-mode-map (if (fboundp 'kbd) (kbd "G x")
				    (read-kbd-macro "G x"))
    'gnus-group-nnimap-expunge))
(add-hook 'gnus-group-mode-hook 'nnimap-group-mode-hook)


;;; Internal functions

;;
;; This is confusing.
;;
;; mark      => read, tick, draft, reply etc
;; flag      => "\\Seen", "\\Flagged", "\\Draft", "gnus-expire" etc
;; predicate => "SEEN", "FLAGGED", "DRAFT", "KEYWORD gnus-expire" etc
;;
;; Mark should not really contain 'read since it's not a "mark" in the Gnus
;; world, but we cheat. Mark == gnus-article-mark-lists + '(read . read).
;;

(defconst nnimap-mark-to-predicate-alist
  (mapcar 
   (lambda (pair) ; cdr is the mark
     (or (assoc (cdr pair)
                '((read . "SEEN")
                  (tick . "FLAGGED")
                  (draft . "DRAFT")
                  (reply . "ANSWERED")))
         (cons (cdr pair)
               (format "KEYWORD gnus-%s" (symbol-name (cdr pair))))))
   (cons '(read . read) gnus-article-mark-lists)))

(defun nnimap-mark-to-predicate (pred)
  "Convert a Gnus mark (a symbol such as read, tick, expire) to a IMAP
predicate (a string such as \"SEEN\", \"FLAGGED\", \"KEYWORD
gnus-expire\") to be used within a IMAP SEARCH query."
  (cdr (assq pred nnimap-mark-to-predicate-alist)))

(defconst nnimap-mark-to-flag-alist 
  (mapcar 
   (lambda (pair)
     (or (assoc (cdr pair)
                '((read . "\\Seen")
                  (tick . "\\Flagged")
                  (draft . "\\Draft")
                  (reply . "\\Answered")))
         (cons (cdr pair)
               (format "gnus-%s" (symbol-name (cdr pair))))))
   (cons '(read . read) gnus-article-mark-lists)))

(defun nnimap-mark-to-flag-1 (preds)
  (if (and (not (null preds)) (listp preds))
      (cons (nnimap-mark-to-flag (car preds))
	    (nnimap-mark-to-flag (cdr preds)))
    (cdr (assoc preds nnimap-mark-to-flag-alist))))

(defun nnimap-mark-to-flag (preds &optional always-list make-string)
  "Convert a Gnus mark (a symbol such as read, tick, expire) to a IMAP
flag (a string such as \"\\Seen\", \"\\Flagged\", \"gnus-expire\") to
be used in a STORE FLAGS command."
  (let ((result (nnimap-mark-to-flag-1 preds)))
    (setq result (if (and (or make-string always-list)
			  (not (listp result)))
		     (list result)
		   result))
    (if make-string
	(mapconcat (lambda (flag)
		     (if (listp flag)
			 (mapconcat 'identity flag " ")
		       flag))
		   result " ")
      result)))

(defun nnimap-mark-permanent-p (mark &optional group)
  "Return t iff MARK can be permanently (between IMAP sessions) saved
on articles, in GROUP."
  (imap-message-flag-permanent-p (nnimap-mark-to-flag mark)))

(defun nnimap-remassoc (key alist)
  "Delete by side effect any elements of LIST whose car is
`equal' to KEY.  The modified LIST is returned.  If the first member
of LIST has a car that is `equal' to KEY, there is no way to remove it
by side effect; therefore, write `(setq foo (remassoc key foo))' to be
sure of changing the value of `foo'."
  (when alist
    (if (equal key (caar alist))
	(cdr alist)
      (setcdr alist (nnimap-remassoc key (cdr alist)))
      alist)))
  
(defun nnimap-update-alist-soft (key value alist)
  (if value 
      (cons (cons key value) (nnimap-remassoc key alist))
    (nnimap-remassoc key alist)))

(defun nnimap-range-to-string (range)
  (mapconcat
   (lambda (item)
     (if (consp item)
         (format "%d:%d" 
                 (car item) (cdr item))
       (format "%d" item)))
   (if (and (listp range) (not (listp (cdr range))))
       (list range) ;; make (1 . 2) into ((1 . 2))
     range)
   ","))

(when nnimap-debug
  (require 'trace)
  (buffer-disable-undo (get-buffer-create nnimap-debug))
  (mapc (lambda (f) (trace-function-background f nnimap-debug))
        '(
nnimap-possibly-change-server
nnimap-verify-uidvalidity
nnimap-find-minmax-uid
nnimap-possibly-change-group
;nnimap-replace-whitespace
nnimap-retrieve-headers-progress
nnimap-retrieve-headers-store-uids
nnimap-retrieve-headers-get-uids
nnimap-retrieve-headers
nnimap-open-connection
nnimap-open-server
nnimap-server-opened
nnimap-close-server
nnimap-request-close
nnimap-status-message
;nnimap-demule
nnimap-request-article-part
nnimap-request-article
nnimap-request-head
nnimap-request-body
nnimap-request-group
nnimap-expunge-close-group
nnimap-close-group
nnimap-pattern-to-list-arguments
nnimap-request-list
nnimap-request-post
nnimap-retrieve-groups
nnimap-request-type
nnimap-request-set-mark
nnimap-split-to-groups
nnimap-split-find-rule
nnimap-split-find-inbox
nnimap-split-articles
nnimap-request-scan
nnimap-request-create-group
nnimap-time-substract
nnimap-date-days-ago
nnimap-request-expire-articles-progress
nnimap-request-expire-articles
nnimap-request-move-article
nnimap-request-accept-article
nnimap-request-delete-group
nnimap-request-rename-group
gnus-group-nnimap-expunge
gnus-group-nnimap-edit-acl
gnus-group-nnimap-edit-acl-done
nnimap-group-mode-hook
nnimap-request-update-info-internal
nnimap-mark-to-predicate
nnimap-mark-to-flag-1
nnimap-mark-to-flag
nnimap-mark-permanent-p
nnimap-update-alist-soft
nnimap-range-to-string
          )))

(provide 'nnimap)

;;; nnimap.el ends here
