;;; nnimap.el --- IMAP backend for Gnus

;;; Copyright (C) 1998 Simon Josefsson <jas@pdc.kth.se>
;;; Copyright (C) 1998 Jim Radford <radford@robby.caltech.edu>
;;; Copyright (C) 1997 John McClary Prevost <visigoth@cs.cmu.edu>

;;; This program is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 2 of the License, or
;;; (at your option) any later version.

;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.

;;; You should have received a copy of the GNU General Public License
;;; along with this program; if not, write to the Free Software
;;; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

;;; Commentary:

;;; This file provides IMAP support for the Emacs news/mailreader Gnus.
;;; To enable this backend you put the following in your .gnus:

;;;  (require  'nnimap)
;;;  (setq gnus-secondary-select-methods 
;;;      '((nnimap 
;;;          "yoyo"
;;;          (nnimap-server-address "robby.caltech.edu"))))

;;; And inside Gnus, use the server buffer ('^', gnus-group-enter-server-mode)
;;; to subscribe to mailboxes. Type '^' in the *Group* buffer, select your
;;; IMAP server and press 'u' to subscribe.

;;;  Note that there is a bug in some versions of gnus-start.el, you
;;;  need to change 
;;;      (while (re-search-backward "[][';?()#]" nil t)
;;;  to
;;;      (while (re-search-backward "[][';?()#.]" nil t)
;;;  otherwise files that start with a dot like .mailboxes (not however
;;;  ~/.mailboxes will give an error.

;;; Todo (in order of priority):

;;;   o nnimap-request-close-group should do incremental flag setting
;;;     instead of first clearing all flags and setting them again. It also
;;;     shouldn't try to set flags at all in read-only groups.
;;;   o 'adaptive nnimap-request-list method?  ('fast, if < x groups => 'slow)
;;;   o Actually use UIDVALIDITY
;;;   o Do we dare to send the EXPUNGE command?
;;;   o Add request-scan with split support
;;;   o fix nnimap-list-pattern
;;;   o Split up big fetches (1,* header especially) in smaller chunks
;;;   o test ephemeral group support
;;;   o use \Draft to support the draft group??
;;;   o What do I do with gnus-newsgroup-*?
;;;   o Figure out when update-info is called and why and if we need to
;;;     call it more often and from where.
;;;   o Add update-info calls? To request-group, close-group, retrieve-groups
;;;   o Add support for the following: (if applicable)

;;;       request-list-newsgroups, request-regenerate
;;;       list-active-group, request-update-mark,
;;;       request-expire-articles,
;;;       request-associate-buffer, request-restore-buffer,
;;;       asynchronous-p

;;;   o When getting a UIDVALIDITY changed message, what do I do to reconstruct
;;;     everything gnus knows about a group?
;;;   o UTF-7 encode mailbox names (does Gnus deal with SPC in group names?)
;;;   o Fix Gnus to handle leading '.' in group names
;;;   o MIME
;;;   o Disconnected support (I haven't a clue about the Gnus Agent thing,
;;;     but IMAP has very good disconnected support)
;;;   

(require 'imap4rev1)

(require 'nnoo)
(require 'nnheader)
(require 'nnmail)
(require 'gnus)
(require 'gnus-range)

(eval-when-compile (require 'cl))

(gnus-declare-backend "nnimap" 'mail 'respool 'address 'prompt-address) ; prompt-address???

(nnoo-declare nnimap) ; we derive from no one

(defvar nnimap-version "nnimap 0.3.10")

(defvoo nnimap-list-pattern "*" 
"*PATTERN or list of PATTERNS use to limit available groups.  

A pattern is either GROUP or (REFERENCE . GROUP).

GROUP is a string.  See the available wildcard characters below.

The meaning of REFERENCE is server-specific, so it's
expected that you (the user) can figure out the appropriate setting.
On the Cyrus server, this is irrelevant.  On the UWash server, this
gets joined together with GROUP.  If it is not specified the
default is an empty string.

Example:
 '(\"INBOX\" \"Mail/*\" \"alt.sex.*\" (\"~friend/Mail/\" . \"list/*\"))

Also note that currently groups that start with a '.' cause Gnus
to choke (a patch is pending acceptance), so instead of something
like \"*\" which might match \".mailboxlist\" you could use
\"~/*\" which would match \"~/.mailboxlist\" and not cause problems.

The two wildcards are * and %.  * means match anything, much like in
shell globbing in Unix.  * does match hierarchy delimiters (. or /, in
the usual case.)  % is the same as *, but does not match the hierarchy
symbol.")

(defvoo nnimap-list-method "LIST" ; "LSUB"
  "*Function called on IMAP server to list groups.  One of \"LSUB\" or
\"LIST\". LSUB means only retrieve groups marked on the server as
subscribed.  LIST means every matching group should be retrieved.")

(defvoo nnimap-server-address nil
  "*The name of the IMAP server.  If nil, uses the virtual server's name.")

(defvoo nnimap-server-port nil
  "*The port of the IMAP server.  If nil, uses the default port.  (143).")

(defvoo nnimap-server-username nil
  "*Login to server with this username.  If nil, interactively ask.")

(defvoo nnimap-server-password nil
  "*Login to server with this password.  If nil, interactively ask.

Do not set this variable unless you fully understand the security 
implications of doing this.")

(defvoo nnimap-split-crosspost t
  "If non-nil, do crossposting if several split methods match the mail.
If nil, the first match found will be used.")

(defvoo nnimap-split-inbox nil
  "*Name of mailbox to split mail from. Mail is read from this mailbox and
split according to rules in nnimap-split-rules.

This can be a string or a list of strings.")

(defvoo nnimap-split-rule nil
  "*Mail will be split according to theese rules. Mail is read from mailbox(es)
specified in nnimap-split-inbox.

If you'd like, for instance, one mail group for mail from the
\"gnus-imap\" mailing list, one group for junk mail and leave
everything else in the incoming mailbox, you could do something like
this:

(setq gnus-secondary-select-methods
      '((nnimap \"mail.server\"
		(nnimap-split-inbox '(\"INBOX\" \"INBOX.sentmail\"))
		(nnimap-split-rule 
		 '((\"INBOX.gnus-imap\"   \"From:.*gnus-imap\")
		   (\"INBOX.junk\"        \"Subject:.*buy\"))))))

As you can see, nnimap-split-rule is a list of lists, where the first
element in each \"rule\" is the name of the IMAP mailbox, and the
second is a regexp that nnimap will try to match on the header to find
a fit.

The first element can also be a list. In that case, the first element
is the server the second element is the group on that server in which
the matching article will be stored.

The second element can also be a function.  In that case, it will be
called narrowed to the headers with the first element of the rule as
the argument.  It should return a non-nil value if it thinks that the
mail belongs in that group.")

(defvoo nnimap-imap-defs nil
  "*Definitions of variables to set up in the IMAP buffer.")

(defvoo nnimap-group-list-speed 'slow ; 'fast, 'medium
  "*If fast, do not show number of articles in the group list.
If medium, guess number of articles by using the UIDNEXT attribute.
If slow, use the exact MESSAGES attribute.")



(defvoo nnimap-group-alist nil)
(defvoo nnimap-server-buffer nil)

;; radford:
;;  shouldn't have to keep this info around, nnoo should???
;;  The only reason for this is `nnimap-server-opened' needs
;;  `nnimap-server-buffer' to do its work, but have to change
;;  servers to get the right one.  I don't think `opened'
;;  should be changing servers.  A better way anyone???
;; jas:
;;  i think this is the way to do it. from "Backend interface" in
;;  the gnus manual:
;;  "When Gnus asks for information from a backend--say nntp---on
;;  something, it will normally include a virtual server name in the
;;  function parameters. (If not, the backend should use the "current"
;;  virtual server.) For instance, nntp-request-list takes a virtual
;;  server as its only (optional) parameter. If this virtual server hasn't
;;  been opened, the function should fail."
(defvar nnimap-server-buffer-alist nil)

(defvar nnimap-debug t)

(when nnimap-debug
  (require 'trace)
  (mapc (lambda (f) (trace-function-background f "*nnimap-debug*")) 
        '(
          nnimap-close-group
          nnimap-close-server
          nnimap-open-server  
          nnimap-possibly-change-group
          nnimap-possibly-change-server
	  nnimap-request-accept-article
          nnimap-request-article
          nnimap-request-article-part
          nnimap-request-body
          nnimap-request-close
          nnimap-request-create-group
          nnimap-request-delete-group
          nnimap-request-group
          nnimap-request-head
          nnimap-request-list
          nnimap-request-list-mapper
          nnimap-request-newgroups
          nnimap-request-post
          nnimap-request-rename-group
          nnimap-request-scan
	  nnimap-split-copy-delete-article
          nnimap-split-move-article
          nnimap-split-to-groups
	  nnimap-split-articles
          nnimap-request-update-info
          nnimap-request-update-mark
          nnimap-retrieve-groups
          nnimap-retrieve-headers
          nnimap-server-opened
          nnimap-status-message
	  nnimap-update-alist-soft
	  nnimap-search
	  nnimap-range-to-string
          nnimap-request-expire-articles
          nnimap-request-move-article
          )))


;;; Interface functions, required backend functions

;;; This seem to define the following:
;;;  `nnimap-close-server'       `nnimap-server-opened'
;;;  `nnimap-open-server'        `nnimap-status-message'
;;; open-server -> nnoo-change-server
;;; close-server -> nnoo-close-server
;;; server-opened -> nnoo-server-opened
;;; status-message -> nnoo-status-message

(nnoo-define-basics nnimap)

(deffoo nnimap-retrieve-headers (articles &optional group server fetch-old)
  (when (nnimap-possibly-change-group group server)
    (with-current-buffer nntp-server-buffer (erase-buffer))
    (let* (compressed (uncompressed (gnus-uncompress-sequence articles)))
      (when (numberp fetch-old) 
        (setcar uncompressed (- (car uncompressed) fetch-old)))
      (setq compressed (gnus-compress-sequence uncompressed t))
      ;; Should we clear `imap-data-folder' ???
      (with-current-buffer nnimap-server-buffer
        (when 
            ;; Should we clear `imap-message-data' before this???  If
            ;; we don't need to cache the headers then I think we
            ;; should.
            (nnimap-ok-p (nnimap-send-command-wait
                          (concat "UID FETCH "
                                  (if (and fetch-old (not (numberp fetch-old)))
                                      "1:*"
                                    (nnimap-range-to-string compressed))
                                  " (UID RFC822.HEADER RFC822.SIZE)")))
          (mapc (lambda (num)
                  (let ((header (imap-message-get num 'RFC822.HEADER))
			(size   (imap-message-get num 'RFC822.SIZE)))
                    (with-current-buffer nntp-server-buffer
                      (if (not header)
                          (insert "423 Bad article number.\n")
                        (insert (format "221 %d Article retrieved.\n" num))
			(insert (format "Chars: %d\n" size))
                        (insert header)
                        (delete-char -1)
                        (insert ".\n")))))
                uncompressed)
          'headers)))))

(deffoo nnimap-open-server (server &optional defs)
  (or (nnimap-server-opened server)
      (progn
        (unless (assq 'nnimap-server-address defs)
          (push (list 'nnimap-server-address server) defs))
        (unless (assq 'nnimap-server-buffer defs)
          (push (list 'nnimap-server-buffer (concat " *nnimap* " server)) defs))
        (nnoo-change-server 'nnimap server defs)
        (condition-case ()
            (when (imap-open-server nnimap-server-address nnimap-server-port
				    nnimap-server-buffer nnimap-imap-defs)
              (imap-send-command "CAPABILITY" nnimap-server-buffer)
              ;; Login
	      (let ((imap-password nnimap-server-password))
		(until (imap-login nnimap-server-username nnimap-server-buffer)
		       (message "Bad Password for %s, Try again." server)
		       (sleep-for 2)))
              (push (cons server nnimap-server-buffer) 
		    nnimap-server-buffer-alist))
	  ('error (kill-buffer nnimap-server-buffer)
		  nil)
          ('quit (let (imap-last-status)
                   (imap-close-server nnimap-server-buffer)
                   (kill-buffer nnimap-server-buffer)
                   nil))))))

(deffoo nnimap-close-server (&optional server)
  (let ((s-b (assoc server nnimap-server-buffer-alist)))
    (when s-b
      (setq nnimap-server-buffer nil)
      (setq nnimap-server-buffer-alist (delq s-b nnimap-server-buffer-alist))
      (imap-close-server (cdr s-b))
      (kill-buffer (cdr s-b)))
    (nnoo-close-server 'nnimap server)))

(deffoo nnimap-request-close ()
  (mapc (lambda (s-b) (nnimap-close-server (car s-b)))
        nnimap-server-buffer-alist)
  (setq nnimap-server-buffer-alist nil))

(deffoo nnimap-server-opened (&optional server)
  (let ((buffer (if server (cdr (assoc server nnimap-server-buffer-alist))
		  nnimap-server-buffer)))
    (if (and (gnus-buffer-live-p buffer)
	     (gnus-buffer-live-p nntp-server-buffer))
	(let ((running (imap-server-opened buffer)))
	  ;; clean up autologouts
	  (unless running
	    (nnimap-close-server server))
	  running))))

;  (and (nnoo-server-opened 'nnimap server)
;       (imap-server-opened nnimap-server-buffer)))

(deffoo nnimap-status-message (&optional server)
  (let ((s-b (assoc server nnimap-server-buffer-alist)))
    (when s-b
      (with-current-buffer (cdr s-b)
	(or (cdr imap-last-status)
	    (nnoo-status-message 'nnimap server))))))

(deffoo nnimap-request-article (article &optional group server to-buffer)
  (nnimap-request-article-part 'RFC822        article group server to-buffer))
(deffoo nnimap-request-head    (article &optional group server to-buffer)
  (nnimap-request-article-part 'RFC822.HEADER article group server to-buffer))
(deffoo nnimap-request-body    (article &optional group server to-buffer)
  (nnimap-request-article-part 'RFC822.TEXT   article group server to-buffer))

(defun nnimap-request-article-part (part article &optional 
					 group server to-buffer)
  (when (nnimap-possibly-change-group group server)
    (with-current-buffer nnimap-server-buffer
      (when (stringp article)
        ;; Find the article number by Message-Id
        (let ((articles (nnimap-search 
			 (format "HEADER Message-Id %d" article))))
          (when (cdr articles)
            (message 
             "IMAP Warning: Message Id search yielded more than one article %s"
             articles))
          (setq article (car articles))))
      (when article
        ;; Find the article by number
        (nnimap-send-command-wait (format "UID FETCH %d (%s)" article part))
        (let ((text (imap-message-get article part)))
          (with-current-buffer (or to-buffer nntp-server-buffer)
            (erase-buffer)
            (if (not text)
                (nnheader-report 
		 'nnimap 
		 (format "Article %s does not exist." article))
              (insert text)
              t)))))))

;;; Select GROUP and unless FAST return 211 EXISTS LOWEST HIGHEST GROUP
;;; The active info returned goes into the `gnus-active-hashtb'.
;;; Don't call this directly, call `gnus-activate-group'.
;;; `gnus-activate-group' can call `request-scan' before request-group.
;;; Ok, here seems to be the problem.  We return 'group and
;;; `gnus-groups-to-gnus-format' seem to not use a prefix "nnimap+yoyo:"
;;; when entering something into the active-hashtb but it does when
;;; it looks for new news.  Damn.
;;;
;;; Also they seem to want us to update the active-info, but
;;; we can;t give a good answer for the lowest message number,
;;; so I think that request-update-info should somehow override
;;; this once it figures out what the real lowest is.
;;;
;;; The 'fast' concept is broken. Gnus =relies= on getting data
;;; back from this function even if called with fast=t. Yuck.
(deffoo nnimap-request-group (group &optional server fast)
  (when (nnimap-possibly-change-group group server) ;; SELECT group
;    (if fast
;	(with-current-buffer nntp-server-buffer 
;	  (erase-buffer)
;	  t)
      (with-current-buffer nnimap-server-buffer
	(let ((imap-message-data (make-vector 2 0))
	      (exists (imap-folder-get 'EXISTS))
	      articles)
	  (if (eq 0 exists)
	      (setq articles '(0))
	    (when (nnimap-ok-p (nnimap-send-command-wait "FETCH 1,* (UID)"))
	      (imap-message-map (lambda (uid Uid)
				  (push uid articles)) 'UID)))
	  (when articles
	    (with-current-buffer nntp-server-buffer
	      (erase-buffer)
	      (insert (format "211 %d %d %d %s\n" 
			      exists (apply 'min articles)
			      (apply 'max articles) group)))
	    t)))));)

(defun gnus-group-normally-qualified (backend server group)
  ;; This is the format for groups in the group-info.
  ;; native groups are not qualified.  Why?
  (let ((method (gnus-server-to-method (format "%s:%s" backend server))))
    (if (gnus-method-equal gnus-select-method method)
        ;; Native (we of course can't call `gnus-native-p' since it
        ;; assumes group is the result of this function
        group
      (gnus-group-prefixed-name group method))))
            
(deffoo nnimap-close-group (group &optional server)
  (when (nnimap-possibly-change-group group server)
    (with-current-buffer nnimap-server-buffer
      ;; For now we assume that we need to sync the group-info
      ;; with the server here.
      (let* ((info (gnus-get-info 
                    (gnus-group-normally-qualified 'nnimap server group)))
             (read (gnus-info-read info))
             (marks (gnus-info-marks info))
             (tick (cdr (assoc 'tick marks)))
             (reply (cdr (assoc 'reply marks))))
	(unless (eq 0 (imap-folder-get 'EXISTS))
	  ;; Cyrus server hangs on empty groups ???
	  (nnimap-send-command-wait 
	   "UID STORE 1:* -FLAGS.SILENT (\\Seen \\Flagged \\Answered)"))
        (mapc
         (lambda (mark)
           (if (car mark)
               (nnimap-send-command-wait 
                (concat "UID STORE " 
                        (nnimap-range-to-string (car mark))
                        " +FLAGS.SILENT (" (cdr mark) ")"))))
         (list (cons read  "\\Seen")
               (cons tick  "\\Flagged")
               (cons reply "\\Answered"))))
      ;; Close her up.  We don't necessarily have to do this.
      (when (nnimap-ok-p (nnimap-send-command-wait "CLOSE"))
        (setq imap-current-folder nil
              imap-message-data nil)
	t))))

;;; Returns: GROUP HIGHEST LOWEST [ynmxj=]
(defun nnimap-request-list-mapper (sym)
      (unless (or (member '\\NoSelect
                          (imap-folder-get 'flags (symbol-name sym)))
                  ;; We ignore groups with spaces (Gnus can't handle them)
                  (string-match " " (symbol-name sym)))
        (let ((group (symbol-name sym)) high)
          (cond 
           ((eq nnimap-group-list-speed 'slow)
	    (when (nnimap-possibly-change-group group nil)
		(let ((imap-message-data (make-vector 2 0))
		      (exists (imap-folder-get 'EXISTS))
		      articles)
		  (if (eq 0 exists)
		      (with-current-buffer nntp-server-buffer
			    (insert (format "%s 0 1 y\n" group)))
		    (when (nnimap-ok-p (nnimap-send-command-wait
					"FETCH 1,* (UID)"))
		      (imap-message-map (lambda (uid Uid) 
					  (push uid articles)) 'UID)
		      (when articles
			(with-current-buffer nntp-server-buffer
			  (insert (format "%s %d %d y\n" group
					  (apply 'max articles)
					  (apply 'min articles))))))))))
           ((eq nnimap-group-list-speed 'medium)
            (gnus-message 10 "STATUS %s" group)
            ;; Need to quote group if it has spaces!!!???
            (when (nnimap-ok-p (nnimap-send-command-wait
                                (concat "STATUS " group 
                                        " (UIDNEXT)")))
              (setq high (imap-folder-get 'UIDNEXT group))
              (with-current-buffer nntp-server-buffer
                (insert (format "%s %d 1 y\n" group (- high 1))))))
           (t ; (eq nnimap-group-list-speed 'fast)
            (with-current-buffer nntp-server-buffer
              (insert (format "%s 0 1 y\n" group))))))))

;;; Returns: GROUP HIGHEST LOWEST [ynmxj=]
(deffoo nnimap-request-list (&optional server)
  (when (nnimap-possibly-change-server server)
    (with-current-buffer nntp-server-buffer (erase-buffer))
    (with-current-buffer nnimap-server-buffer
      (setq imap-data-folder (make-vector imap-convenient-group-prime 0)
	    imap-current-folder nil)
      (let* ((patterns (mapcar (lambda (p) (if (consp p) p (cons "\"\"" p)))
                               (if (or (atom nnimap-list-pattern)
                                       (atom (cdr nnimap-list-pattern)))
                                   (list nnimap-list-pattern)
                                 nnimap-list-pattern)))
             pattern)
        (while (setq pattern (pop patterns))
          (nnimap-ok-p (nnimap-send-command-wait
                        (concat nnimap-list-method " "
                                (car pattern) " "
                                (cdr pattern)))))
        (gnus-message 8 "Generating active list")
        (mapatoms 'nnimap-request-list-mapper imap-data-folder)))
    (setq nnimap-group-alist (nnmail-get-active))
    t))
  

;;; IMAP doesn't support posting, but this must be defined
(deffoo nnimap-request-post (&optional server)
  nil)


;;; Interface functions, optional backend functions


;; Note that request-scan gets called right before this
;; from `read-active-file'.
;; Switch to active format to convey READ-ONLY status???
(deffoo nnimap-retrieve-----groups (groups &optional server)
  ;; comment this function out ----- until we see if it is needed, since
  ;; it really doesn't provide accurate info
    (with-current-buffer nntp-server-buffer (erase-buffer))
    (when (nnimap-possibly-change-server server)
      (with-current-buffer nnimap-server-buffer
        (while groups
          (when (nnimap-send-command-wait (concat "STATUS " (car groups)
                                                " (MESSAGES UIDNEXT)"))
            (let* ((uidnext (imap-folder-get 'UIDNEXT (car groups)))
                   (messages (imap-folder-get 'MESSAGES (car groups))))
              (with-current-buffer nntp-server-buffer
                ;; What should I return on empty???
                (if messages
                    (insert (format "211 %d 1 %d %s\n" 
                                messages (- uidnext 1) (car groups)))))))
          (pop groups)))
      'group))

;;; I really think this should update the active-info too???
(deffoo nnimap-request-update-info (group info &optional server)
  (when (nnimap-possibly-change-group group server)
    (with-current-buffer nnimap-server-buffer
      ;; extend info to have parameters (otherwise when we set them, 
      ;; they don't get set)
      (or (gnus-info-params info) 
	  (gnus-info-set-params info nil t))
      
      ;; As long as the UIDVALIDITY is the same, UIDs
      ;; are vaild and persist (so save it)
      (gnus-group-set-parameter group 'uidvalidity
                                (imap-folder-get 'uidvalidity))
      ;; Replace list of read and marked articles with authoritative
      ;; data from server
      (gnus-info-set-read 
       info 
       ;; Heavy optimizations are possible if someone answered the following:
       ;;
       ;; does UID SEARCH returned sorted data? (probably not)
       ;; does gnus-uncompress-range return sorted data? (probably?)
       ;; does gnus-sorted-complement return sorted data? (probably?)
       ;; does gnus-compress-sequence handle unsorted data? (probably not?)
       ;;
       ;; Other improvements would be available if someone wrote
       ;; a range version of gnus-sorted-complement.
       (let* ((oldread (gnus-info-read info))
	      (seen (nnimap-search "SEEN"))
	      (seens (sort seen '<))
	      (unseen (nnimap-search "UNSEEN"))
	      (unseens (sort unseen '<))
	      (oldandnewseentmp (gnus-add-to-range oldread seens))
	      (oldandnewseen (gnus-uncompress-range oldandnewseentmp))
	      (oldandnewseens (sort oldandnewseen '<))
	      (correctedtmp (gnus-sorted-complement oldandnewseens unseens))
	      (correctedtmps (sort correctedtmp '<))
	      (corrected (gnus-compress-sequence correctedtmps t)))
	 corrected))
      (gnus-info-set-marks 
       info 
       (nnimap-update-alist-soft 'tick (gnus-compress-sequence 
                                        (nnimap-search "FLAGGED"))
                                 (gnus-info-marks info)))
      (gnus-info-set-marks
       info 
       (nnimap-update-alist-soft 'reply (gnus-compress-sequence 
                                         (nnimap-search "ANSWERED"))
                                 (gnus-info-marks info))))
    info))

;;; Respond to articles with mail
(deffoo nnimap-request-type (group article)
  'mail)

;; Move article ARTICLE from group GROUP on current server to group
;; TO-GROUP.  
(defun nnimap-split-copy-delete-article (article group to-group)
  (when (nnimap-ok-p (nnimap-send-command-wait  ;;  xxx UID COPY is not std
		      (format "UID COPY %d %s" article to-group))) 
    (let ((store 
	   (nnimap-send-command-wait 
	    (format "UID STORE %d +FLAGS (\\Flagged)" article))))
      (if (nnimap-ok-p store)
	  (message "IMAP split: moved %s:%d to %s" group article to-group)
	(error "IMAP flag store failed: %s (you may have unread mail marked as read!)" store)))))

(defun nnimap-split-move-article (article group to-group)
  (when to-group
    (unless (nnimap-split-copy-delete-article article group to-group)
      (message "Could not find mailbox %s, creating..." to-group)
      (if (nnimap-ok-p (nnimap-send-command-wait
			(format "CREATE %s" to-group)))
	  (nnimap-split-copy-delete-article article group to-group)
	(message "Could not create mailbox %s: %s" 
		 to-group imap-last-status)))))

;; doesn't use COPY but instead FETCH+APPEND. Slower.
(defun nnimap-split--move-article (article group to-group)
  (when to-group
    (message "IMAP split: moving %s:%d to %s" group article to-group)
    
    ;; mailbox exist?  create it or bail out
    (with-current-buffer nnimap-server-buffer (setq imap-data-folder [0]))
    (when (and (nnimap-ok-p (nnimap-send-command-wait 
			     (concat "LIST \"\" " to-group)
			     nnimap-server-buffer))
	       (if (imap-folder-get 'delimiter to-group 
				    nnimap-server-buffer)
		   t
		 (if (nnimap-ok-p (nnimap-send-command-wait
				   (concat "CREATE " to-group)
				   nnimap-server-buffer))
		     t
		   (progn 
		     (message "Couldn't create group %s, leaving article in %s"
			      to-group group)
		     nil))))
      ;; ok, group exist, move article
      (unless (nnimap-request-move-article 
	       article group nil
	       (list 'nnimap-request-accept-article to-group))
	(message "Couldn't move article %s:%d to %s" group article to-group)
	;; remove Seen flag on article -- request-move-article fetches
	;; the whole article, which set's this flag. Bad.
	(unless (nnimap-ok-p 
		 (nnimap-send-command-wait 
		  (format "UID STORE %d -FLAGS.SILENT \\Seen" article)))
	  ;; serious damage
	  (error "Couldn't remove Seen flag on article %s:%d (you might have unread mail flagged as read!)" group article))))))

;; tries to match all rules in nnimap-split-rule against content of 
;; nntp-server-buffer, returns a list of groups that matched. List will 
;; contain nil, which should be disregarded. (todo: fix nil's)
(defun nnimap-split-to-groups ()
  (with-current-buffer nntp-server-buffer
    (mapcar (lambda (rule) 
	      (let ((group (car rule))
		    (regexp (cadr rule)))
		(goto-char (point-min))
		(when (re-search-forward regexp nil t)
		  group)))
	    (with-current-buffer nnimap-server-buffer nnimap-split-rule))))

;; 47
(defun nnimap-split-articles (&optional group server)
  (when (nnimap-possibly-change-server server)
    (with-current-buffer nnimap-server-buffer
      (let (inbox (inboxes (if (atom nnimap-split-inbox)
			       (list nnimap-split-inbox)
			     nnimap-split-inbox)))
	;; iterate over inboxes
	(while (and (setq inbox (pop inboxes))
		    (nnimap-possibly-change-group inbox))
	  (let (article (unseens (nnimap-search "UNSEEN")))
	    ;; iterate over articles
	    (while (setq article (pop unseens))
	      (when (nnimap-request-head article)
		;; article into what groups?
		(let ((groups (nnimap-split-to-groups)))
		  ;; move it there
		  (if nnimap-split-crosspost
		      ;; move to all boxes
		      (let (to-group)
			(while (setq to-group (pop groups))
			  (nnimap-split-move-article article inbox to-group)))
		    ;; move to first non-nil box
		    (let ((group (until (pop groups))))
		      (nnimap-split-move-article article inbox 
						 group))))))))))))

;; nnimap-request-scan doesn't need to do anything. the delivery agent
;; program is responsible for putting new message in the imap folders.
;; compare the situation with nntp (nil function) and with all nnmail-
;; backends (fetches mail from spools, POPing, saving in files on local
;; storage etc).
;; On the other hand, we could do message splitting here.
(deffoo nnimap-request-scan (&optional group server)
  (nnimap-split-articles group server))

;; (deffoo nnimap-request-group-description

;; (deffoo nnimap-request-list-newsgroups

;; For some reason this must be defined.  Why???  ANSWER: Bug in Gnus, fixed.
(deffoo nnimap-request-newgroups (date &optional server)
  (when (nnimap-possibly-change-server server)
    (with-current-buffer nntp-server-buffer
      (erase-buffer))
    t))
    
;; (deffoo nnimap-request-update-mark (group article mark)

;;; Maybe we should add to current group-alist???
(deffoo nnimap-request-create-group (group &optional server args)
  (when (nnimap-possibly-change-server server)
    (nnimap-ok-p (nnimap-send-command-wait
                  (concat "CREATE " group) 
                  nnimap-server-buffer))))

;;; Notice that we don't actually delete anything, we just mark them
;;; deleted.  We need some kind of EXPUNGE command.  Also, when we
;;; expunge we delete all marked \Deleted articles.  So we either
;;; need to unmark,expunge,re-mark or come up with some new scheme.
;;; I guess force means delete, not force means expire.  We don't to
;;; any expiry currently.
;;; Also, we can search for articles older than some expiry time if
;;; we need to.  I'm not sure what to do.
(deffoo nnimap-request-expire-articles 
    (articles group &optional server force)
  (when force
    (when (nnimap-possibly-change-group group server)
      (with-current-buffer nnimap-server-buffer
;;;        (setq deleted (nnimap-search "DELETED"))
;;;        (nnimap-ok-p (nnimap-send-command-wait "EXPUNGE"))
;;;        (setq articles (gnus-sorted-complement articles deleted)))))
        (when (nnimap-ok-p (nnimap-send-command-wait 
                            (concat "UID STORE "
                                    (nnimap-range-to-string
                                     (gnus-compress-sequence articles))
                                    " +FLAGS (\\Deleted)")))
          (setq articles nil)))))
  ;; return articles not deleted
  articles)

(deffoo nnimap-request-move-article (article group server
					     accept-form &optional last)
  (save-excursion
    (let ((buf (get-buffer-create " *nnimap move*"))
	  result)
      (and 
       (nnimap-request-article article group server)
       (save-excursion
	 (set-buffer buf)
	 (buffer-disable-undo (current-buffer))
	 (insert-buffer-substring nntp-server-buffer)
	 (setq result (eval accept-form))
	 (kill-buffer buf)
	 result)
       (nnimap-request-expire-articles (list article) group server t))
      result)))

;;; Maybe we should update the group-alist???
(deffoo nnimap-request-accept-article (group &optional server last)
  (when (nnimap-possibly-change-server server)
    ;; The article will be appended as UIDNEXT
    (when (nnimap-ok-p (nnimap-send-command-wait
			(concat "STATUS " group " (UIDNEXT)")
			nnimap-server-buffer))
      (with-current-buffer (current-buffer)
	(goto-char (point-min))
	(unless (string= "\n" imap-eol)
	  (while (re-search-forward "\n" nil t)
	    (replace-match imap-eol))))
      (nnimap-ok-p (nnimap-send-command-wait 
		    ;; Optional flags,date???
		    (list (format "APPEND %s " group) (current-buffer))
		    nnimap-server-buffer))
      (let ((high (imap-folder-get 'UIDNEXT group nnimap-server-buffer)))
	(when high
	  (cons group high))))))

;; (deffoo nnimap-request-replace-article -- IMAP does not support replacing

;;; Maybe we should remove from group-alist and unset if current group???
(deffoo nnimap-request-delete-group (group force &optional server)
  (when (nnimap-possibly-change-server server)
    (nnimap-ok-p (nnimap-send-command-wait
                  (concat "DELETE " group)
                  nnimap-server-buffer))))

;;; Maybe we should update the group-alist and current group???
(deffoo nnimap-request-rename-group (group new-name &optional server)
  (when (nnimap-possibly-change-server server)
    (when (string= "INBOX" group)
      (nnheader-report 
       'nnimap 
       "Renaming INBOX will only move content to a newly created mailbox")
      (sit-for 1))
    (nnimap-ok-p (nnimap-send-command-wait
                  (concat "RENAME " group " " new-name)
                  nnimap-server-buffer))))


;;; Internal functions

(defun nnimap-update-alist-soft (key value alist)
  (if value 
      (cons (cons key value) (remassoc key alist))
    (remassoc key alist)))

(defun nnimap-range-to-string (range)
  (mapconcat 
   (lambda (item)
     (if (consp item)
         (format "%d:%d" 
                 (car item) (cdr item))
       (format "%d" item)))
   range
   ","))

(defun nnimap-send-command-wait (command &optional buffer)
  (gnus-message 10 (apply 'concat
                         (if (listp command)
                             (mapcar
                              (lambda (s-b) 
                                (if (bufferp s-b) (buffer-name s-b) s-b)) 
                              command)
                           (list command))))
  (imap-send-command-wait command buffer))

(defun nnimap-ok-p (status)
  (if status
      (or (eq 'OK (car status))
          (nnheader-report 'nnimap (cdr status)))
    (nnheader-report 'nnimap (format "IMAP Command Timed Out"))))

; Shouldn't this already exist???
(defmacro until (test &rest body)
  "Execute BODY until TEST returns something non nil and return that."
  (let ((test-return (make-symbol "test-return")))
    `(let ((,test-return))
       (while (not (setq ,test-return ,test))
         ,@body)
       ,test-return)))
(put 'until 'lisp-indent-function (get 'while 'lisp-indent-function))

(defun nnimap-search (predicate)
  (imap-folder-set 'search nil)
  (when (nnimap-ok-p (nnimap-send-command-wait 
		      (concat "UID SEARCH " predicate)))
    (imap-folder-get 'search)))

(defun nnimap-possibly-change-server (server)
  "Change to server SERVER if needed (open it if it's closed). If SERVER is
nil, change to current server."
  (and (nnimap-open-server (nnoo-current-server 'nnimap))
       (nnoo-change-server 'nnimap (or server (nnoo-current-server 'nnimap))
			   nil)))

(defun nnimap-possibly-change-group (group &optional server)
  (when (nnimap-possibly-change-server (or server (nnoo-current-server 
						   'nnimap)))
    (with-current-buffer nnimap-server-buffer
      (when group
	(unless (string= group imap-current-folder)
	  (setq imap-message-data (make-vector 199 0)) ; itty bitty obarray???
	  ;; Folder must be set during SELECT
	  (setq imap-current-folder group)
	  (unless (nnimap-ok-p (nnimap-send-command-wait
				(concat (if (gnus-ephemeral-group-p group)
					    "EXAMINE"
                                        "SELECT")
					" " group)))
	    ;; Failed SELECT unselects the current group
	    (setq imap-current-folder nil))))
      imap-current-folder)))

(provide 'nnimap)