;; nnimap.el --- IMAP backend for Gnus

;; Copyright (C) 1998 Simon Josefsson
;; Copyright (C) 1998 Jim Radford
;; Copyright (C) 1997 John McClary Prevost

;; This file is part of Emacs.

;; Emacs is free software; you can redistribute it and/or modify it
;; under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; Emacs is distributed in the hope that it will be useful, but
;; WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;; General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with XEmacs; see the file COPYING.  If not, write to the Free
;; Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
;; 02111-1307, USA.

;;; Commentary:

;;; This file provides IMAP support for the Emacs news/mailreader Gnus.
;;; To enable this backend you can put something like the following in
;;; your .gnus file:

;;;  (require  'nnimap)
;;;  (setq gnus-secondary-select-methods 
;;;      '((nnimap 
;;;          "yoyo"
;;;          (nnimap-server-address "robby.caltech.edu")
;;;          (nnimap-list-pattern  '("INBOX" "Mail/*" "alt.sex.*")))))

;;; Or

;;;  Inside Gnus, use:
;;;          ^ (gnus-group-enter-server-mode)
;;;          a (gnus-server-add-server)
;;;    Server method:
;;;          nnimap
;;;    Server name:
;;;          mail1.andrew.cmu.edu
;;;
;;;  Move to the folders you wish to read as groups in Gnus and press u
;;;   on each to subscribe you to them in Gnus.

;;;  Note that there is a bug in some versions of gnus-start.el, you
;;;  need to change 
;;;      (while (re-search-backward "[][';?()#]" nil t)
;;;  to
;;;      (while (re-search-backward "[][';?()#.]" nil t)
;;;  otherwise files that start with a dot like .mailboxes (not however
;;;  ~/.mailboxes will give an error.

;;; Todo:

;;;   o test ephemeral group support
;;;   o use \Draft to support the draft group??
;;;   o Actually use UIDVALIDITY
;;;   o Figure out what -scan is really supposed to do
;;;   o Figure out when update-info is called and why and if we need to
;;;     call it more often and from where.
;;;   o Send patch in for gnus-start.el to fix groups starting with '.'
;;;   o Expire/Delete support?
;;;   o What do I do with gnus-newsgroup-*?
;;;   o Guess: Add update-info calls to request-group, close-group, retrieve-groups
;;;   o Add support for the following: (if applicable)

;;;       request-list-newsgroups, request-regenerate
;;;       list-active-group, request-type, request-update-mark,
;;;       request-post, request-scan, request-expire-articles,
;;;       request-move-article, request-replace-article,
;;;       request-associate-buffer, request-restore-buffer,
;;;       assyncronous-p

;;;   o What do I return when there are zero articles?   ANSWER: high=0 low=1
;;;   o When get a UIDVALIDITY changed message, what do I do to reconstruct
;;;     everything gnus knows about a group?
;;;   o We need to quote groups with spaces (does Gnus deal?)
;;;   o Never use "*" in LIST command, is % instead (as Netscape does)

;;; $Id: nnimap.el,v 1.14.1.4 1998/06/21 23:45:54 jas Exp $

(require 'imap4rev1)

(require 'nnoo)
(require 'nnheader)
(require 'gnus-range)
(require 'nnmail)
(require 'gnus)

(eval-when-compile (require 'cl))

;(eval-and-compile
;  (autoload 'nnmail-read-passwd "nnmail"))

(gnus-declare-backend "nnimap" 'mail 'respool 'address 'prompt-address) ; prompt-address???

(nnoo-declare nnimap) ; we derive from no one

(defvar nnimap-version "0.3.2")

(defvoo nnimap-list-pattern "*" 
"*PATTERN or list of PATTERNS use to limit available groups.  

A pattern is either GROUP or (REFERENCE . GROUP).

GROUP is a string.  See the available wildcard characters below.

The meaning of REFERENCE is server-specific, so it's
expected that you (the user) can figure out the appropriate setting.
On the Cyrus server, this is irrelevant.  On the UWash server, this
gets joined together with GROUP.  If it is not specified the
default is an empty string.

Example:
 '(\"INBOX\" \"Mail/*\" \"alt.sex.*\" (\"~friend/Mail/\" . \"list/*\"))

Also note that currently groups that start with a '.' cause Gnus
to choke (a patch is pending acceptance), so instead of something
like \"*\" which might match \".mailboxlist\" you could use
\"~/*\" which would match \"~/.mailboxlist\" and not cause problems.

The two wildcards are * and %.  * means match anything, much like in
shell globbing in Unix.  * does match hierarchy delimiters (. or /, in
the usual case.)  % is the same as *, but does not match the hierarchy
symbol.")

(defvoo nnimap-list-method "LIST" ; "LSUB"
  "*Function called on IMAP server to list groups.  One of \"LSUB\" or
\"LIST\". LSUB means only retrieve groups marked on the server as
subscribed.  LIST means every matching group should be retrieved.")

(defvoo nnimap-server-address nil
  "*The name of the IMAP server.  If nil, uses the virtual server's name.")

(defvoo nnimap-server-port nil
  "*The port of the IMAP server.  If nil, uses the default port.  (143).")

(defvoo nnimap-imap-defs nil
  "*Definitions of variables to set up in the IMAP buffer.")

(defvoo nnimap-group-list-speed 'slow ; 'fast, 'medium
  "*If fast, do not show number of articles in the group list.
If medium, guess number of articles by using the UIDNEXT attribute.
If slow, use the exact MESSAGES attribue.")

(defvoo nnimap-group-alist nil)
(defvoo nnimap-server-buffer nil)



(defun nnimap-send-command-wait (command &optional buffer)
  (gnus-message 5 (apply 'concat
                         (if (listp command)
                             (mapcar
                              (lambda (s-b) 
                                (if (bufferp s-b) (buffer-name s-b) s-b)) 
                              command)
                           (list command))))
  (imap-send-command-wait command buffer))

(defun nnimap-ok-p (status)
  (if status
      (or (eq 'OK (car status))
          (nnheader-report 'nnimap (cdr status)))
    (nnheader-report 'nnimap (format "IMAP Command Timed Out"))))

; Shouldn't this already exist???
(defmacro until (test &rest body)
  "Execute BODY until TEST returns something non nil and return that."
  (let ((test-return (make-symbol "test-return")))
    `(let ((,test-return))
       (while (not (setq ,test-return ,test))
         ,@body)
       ,test-return)))
(put 'until 'lisp-indent-function (get 'while 'lisp-indent-function))

(defvar nnimap-debug t)

(when nnimap-debug
  (require 'trace)
  (mapc (lambda (f) (trace-function-background f "*nnimap-debug*")) 
        '(
          nnimap-close-group
          nnimap-close-server
          nnimap-open-server
          nnimap-possibly-change-group
          nnimap-request-accept-article
          nnimap-request-article
          nnimap-request-article-part
          nnimap-request-body
          nnimap-request-close
          nnimap-request-create-group
          nnimap-request-delete-group
          nnimap-request-group
          nnimap-request-head
          nnimap-request-list
          nnimap-request-newgroups
          nnimap-request-post
          nnimap-request-rename-group
          nnimap-request-scan
          nnimap-request-update-info
          nnimap-request-update-mark
          nnimap-retrieve-groups
          nnimap-retrieve-headers
          nnimap-server-opened
          nnimap-status-message
	  nnimap-update-alist-soft
	  nnimap-search
	  nnimap-range-to-string

;          nnimap-active-group
;          nnimap-assyncronous-p
;          nnimap-request-associate-buffer
          nnimap-request-expire-articles
;          nnimap-request-move-article
;          nnimap-request-post
;          nnimap-request-regenerate
;          nnimap-request-replace-article
;          nnimap-request-restore-buffer
;          nnimap-request-scan
;          nnimap-request-type
;          nnimap-request-update-mark
          )))

;;; Interface functions.

;;; This seem to define the following:
;;;  `nnimap-close-server'       `nnimap-server-opened'
;;;  `nnimap-open-server'        `nnimap-status-message'
;;; open-server -> nnoo-change-server
;;; close-server -> nnoo-close-server
;;; server-opened -> nnoo-server-opened
;;; status-message -> nnoo-status-message
;(nnoo-define-basics nnimap) 

(defun nnimap-range-to-string (range)
  (mapconcat 
   (lambda (item)
     (if (consp item)
         (format "%d:%d" 
                 (car item) (cdr item))
       (format "%d" item)))
   (if (and (listp range) (listp (cdr range))) range (list range))
   ","))

;; We can add RFC822.SIZE easily, but Gnus wants Lines: but doesn't use
;; Chars:.
(deffoo nnimap-retrieve-headers (articles &optional group server fetch-old)
  (when (nnimap-possibly-change-group group server)
    (with-current-buffer nntp-server-buffer (erase-buffer))
    (let* (compressed (uncompressed (gnus-uncompress-sequence articles)))
      (when (numberp fetch-old) 
        (setcar uncompressed (- (car uncompressed) fetch-old)))
      (setq compressed (gnus-compress-sequence uncompressed t))
      ;; Should we clear `imap-data-folder' ???
      (with-current-buffer nnimap-server-buffer
        (when 
            ;; Should we clear `imap-message-data' before this???  If
            ;; we don't need to cache the headers then I think we
            ;; should.
            (nnimap-ok-p (nnimap-send-command-wait
                          (concat "UID FETCH "
                                  (if (and fetch-old (not (numberp fetch-old)))
                                      "1:*" ;; XXX not nice, break it up instead
                                    (nnimap-range-to-string compressed))
                                  " (UID RFC822.HEADER)")))
          (mapc (lambda (num)
                  (let ((header
                         (imap-message-get
                          num
                          'RFC822.HEADER)))
                    (with-current-buffer nntp-server-buffer
                      (if (not header)
                          (insert "423 Bad article number.\n")
                        (insert (concat "221 " num " Article retrieved.\n"))
                        (insert header)
                        (delete-char -1)
                        (insert ".\n")))))
                uncompressed)
          'headers)))))

;; radford:
;;  shouldn't have to keep this info around, nnoo should???
;;  The only reason for this is `nnnimap-server-opened' needs
;;  `nnimap-server-buffer' to do its work, but have to change
;;  servers to get the right one.  I don't think `opened'
;;  should be changing servers.  A better way anyone???
;; jas:
;;  i think this is the way to do it. from "Backend interface" in
;;  the gnus manual:
;;  "When Gnus asks for information from a backend--say nntp---on
;;  something, it will normally include a virtual server name in the
;;  function parameters. (If not, the backend should use the "current"
;;  virtual server.) For instance, nntp-request-list takes a virtual
;;  server as its only (optional) parameter. If this virtual server hasn't
;;  been opened, the function should fail."

(defvar nnimap-server-buffer-alist nil)

(deffoo nnimap-server-opened (&optional server)
  (let ((buffer (if server (cdr (assoc server nnimap-server-buffer-alist))
		  nnimap-server-buffer)))
  (and (gnus-buffer-live-p buffer)
       (gnus-buffer-live-p nntp-server-buffer)
       (imap-server-opened buffer))))
;  (and (nnoo-server-opened 'nnimap server)
;       (imap-server-opened nnimap-server-buffer)))

(deffoo nnimap-open-server (server &optional defs)
  (or (nnimap-server-opened server)
      (progn
;        (push (list 'nnimap-server-name server) defs)
        (unless (assq 'nnimap-server-address defs)
          (push (list 'nnimap-server-address server) defs))
        (unless (assq 'nnimap-server-buffer defs)
          (push (list 'nnimap-server-buffer (concat " *nnimap* " server)) defs))
        (nnoo-change-server 'nnimap server defs)
        (if (condition-case ()
		(imap-open-server nnimap-server-address nnimap-server-port
				  nnimap-server-buffer nnimap-imap-defs)
	      (quit nil)
	      (error nil))
	    (progn
	      (push (cons server nnimap-server-buffer) nnimap-server-buffer-alist)
	      (with-current-buffer nnimap-server-buffer
		(nnimap-send-command-wait "CAPABILITY"))
	      ;; Login
	      (condition-case ()
		  (until (imap-login nil nnimap-server-buffer)
			 (message "Bad Password for %s, Try again." server)
			 (sleep-for 2))
		('quit (let (imap-last-status)
			 (imap-close-server nnimap-server-buffer)
			 (kill-buffer nnimap-server-buffer)
			 nil))))
	  (progn
	    (imap-close-server nnimap-server-buffer)
	    (kill-buffer nnimap-server-buffer)
	    nil)))))

(deffoo nnimap-close-server (&optional server)
  (let ((s-b (assoc server nnimap-server-buffer-alist)))
    (when s-b
      (setq nnimap-server-buffer-alist (delq s-b nnimap-server-buffer-alist))
      (imap-close-server (cdr s-b))
      (kill-buffer (cdr s-b)))
    (nnoo-close-server 'nnimap server)))

(deffoo nnimap-request-close ()
  (mapc (lambda (s-b) (nnimap-close-server (car s-b)))
        nnimap-server-buffer-alist)
  (setq nnimap-server-buffer-alist nil))

(deffoo nnimap-status-message (&optional server)
  (let ((s-b (assoc server nnimap-server-buffer-alist)))
    (when s-b
      (with-current-buffer s-b
	(or (cdr imap-last-status)
	    (nnoo-status-message 'nnimap server))))))

(defun nnimap-search (predicate)
  (imap-folder-set 'search nil)
  (when (nnimap-ok-p (nnimap-send-command-wait (concat "UID SEARCH " predicate)))
    (imap-folder-get 'search)))
    
(deffoo nnimap-request-article (article &optional group server to-buffer)
  (nnimap-request-article-part 'RFC822        article group server to-buffer))
(deffoo nnimap-request-head    (article &optional group server to-buffer)
  (nnimap-request-article-part 'RFC822.TEXT   article group server to-buffer))
(deffoo nnimap-request-body    (article &optional group server to-buffer)
  (nnimap-request-article-part 'RFC822.HEADER article group server to-buffer))

(defun nnimap-request-article-part (part article &optional group server to-buffer)
  (when (nnimap-possibly-change-group group server)
    (with-current-buffer nnimap-server-buffer
      (when (stringp article)
        ;; Find the article number by Message-Id
        (let ((articles (nnimap-search (concat "HEADER Message-Id " article))))
          (when (cdr articles)
            (message 
             "IMAP Warning: Message Id search yeiled more than one article %s"
             articles))
          (setq article (car articles))))
      (when article
        ;; Find the article by number
        (nnimap-send-command-wait (format "UID FETCH %d (%s)" article part))
        (let ((text (imap-message-get article part)))
          (with-current-buffer (or to-buffer nntp-server-buffer)
            (erase-buffer)
            (if (not text)
                (nnheader-report 'nnimap (format "Article %s does not exist." article))
              (insert text)
              t)))))))

;; Note that request-scan gets called right before this
;; from `read-active-file'.
;; Switch to active format to convey READ-ONLY status???
(deffoo nnimap-retrieve-----groups (groups &optional server)
  ;; comment this function out ----- until we see if it is needed, since
  ;; it really doesn't provied accurate info
    (with-current-buffer nntp-server-buffer (erase-buffer))
    (when (nnimap-possibly-change-server server)
      (with-current-buffer nnimap-server-buffer
        (while groups
          (when (nnimap-send-command-wait (concat "STATUS " (car groups)
                                                " (MESSAGES UIDNEXT)"))
            (let* ((uidnext (imap-folder-get 'UIDNEXT (car groups)))
                   (messages (imap-folder-get 'MESSAGES (car groups))))
              (with-current-buffer nntp-server-buffer
                ;; What should I return on empty???
                (if messages
                    (insert (format "211 %d 1 %d %s\n" 
                                messages (- uidnext 1) (car groups)))))))
          (pop groups)))
      'group))

;;; Select GROUP and unless FAST return 211 EXISTS LOWEST HIGHEST GROUP
;;; The active info returned goes into the `gnus-active-hashtb'.
;;; Don't call this directly, call `gnus-activate-group'.
;;; `gnus-activate-group' can call `request-scan' before request-group.
;;; Ok, here seems to be the problem.  We return 'group and
;;; `gnus-groups-to-gnus-format' seem to not use a prefix "nnimap+yoyo:"
;;; when entering something into the active-hashtb but it does when
;;; it looks for new news.  Damn.
;;;
;;; Also they seem to want us to update the active-info, but
;;; we can;t give a good answer for the lowest message number,
;;; so I think that request-update-info should somehow override
;;; this once it figures out what the real lowest is.
(deffoo nnimap-request-group (group &optional server fast)
  (when (nnimap-possibly-change-group group server) ;; SELECT group
    (with-current-buffer nnimap-server-buffer
      (if nil;fast
          t
        (if nil ;; STATUS is slower than FETCH since we have the group SELECTED
            (when (nnimap-ok-p (nnimap-send-command-wait
                                (concat "STATUS " group
                                        " (MESSAGES UIDNEXT)")))
              (let* ((messages (imap-folder-get 'MESSAGES group))
                     (uidnext (imap-folder-get 'UIDNEXT group)))
                (when messages
                  (with-current-buffer nntp-server-buffer
                    (erase-buffer)
                    (insert (concat "211 %d 1 %d %s\n" 
                                    messages (- uidnext 1) group))
                    t))))
          (let ((imap-message-data (make-vector 2 0))
                (exists (imap-folder-get 'EXISTS))
                articles)
            (if (eq 0 exists)
                (setq articles '(0))
              (when (nnimap-ok-p (nnimap-send-command-wait
                                  (concat "FETCH 1,* (UID)")))
                (imap-message-map (lambda (uid Uid) (push uid articles)) 
                                  'UID)))
            (when articles
              (with-current-buffer nntp-server-buffer
                (erase-buffer)
                (insert (format "211 %d %d %d %s\n" 
                                exists (apply 'min articles)
                                (apply 'max articles) group)))
              t)))))))


(defun gnus-group-normally-qualified (backend server group)
  ;; This is the format for groups in the group-info.
  ;; native groups are not qualified.  Why?
  (let ((method (gnus-server-to-method (format "%s:%s" backend server))))
    (if (gnus-method-equal gnus-select-method method)
        ;; Native (we of course can't call `gnus-native-p' since it
        ;; assumes group is the result of this function
        group
      (gnus-group-prefixed-name group method))))
            
(deffoo nnimap-close-group (group &optional server)
  (when (nnimap-possibly-change-group group server)
    (with-current-buffer nnimap-server-buffer
      ;; For now we assume that we need to sync the group-info
      ;; with the server here.
      (let* ((info (gnus-get-info 
                    (gnus-group-normally-qualified 'nnimap server group)))
             (read (car (gnus-info-read info)))
             (marks (gnus-info-marks info))
             (tick (cdr (assoc 'tick marks)))
             (reply (cdr (assoc 'reply marks))))
	(unless (eq 0 (imap-folder-get 'EXISTS))
	  (nnimap-send-command-wait 
	   "UID STORE 1:* -FLAGS.SILENT (\\Seen \\Flagged \\Answered)"))
        (mapc
         (lambda (mark)
           (if (car mark)
               (nnimap-send-command-wait 
                (concat "UID STORE " 
                        (nnimap-range-to-string (car mark))
                        " +FLAGS.SILENT (" (cdr mark) ")"))))
         (list (cons read  "\\Seen")
               (cons tick  "\\Flagged")
               (cons reply "\\Answered"))))
      ;; Close her up.  We don't necessarily have to do this.
      (when (nnimap-ok-p (nnimap-send-command-wait "CLOSE"))
        (setq imap-current-folder nil
              imap-message-data nil))
      t)))

;; For some reason this must be defined.  Why???
(deffoo nnimap-request-newgroups (date &optional server)
  (when (nnimap-possibly-change-server server)
    (with-current-buffer nntp-server-buffer
      (erase-buffer))
    nil)) ;t

(setq nnimap-request-list-mapper 
      (lambda (sym)
	(unless (or (member '\\NoSelect
			    (imap-folder-get 'flags (symbol-name sym)))
		    ;; We ignore groups with spaces (Gnus can't handle them)
		    (string-match " " (symbol-name sym)))
	  (let ((group (symbol-name sym)) high)
	    (cond 
	     ((eq nnimap-group-list-speed 'slow)
	      (when (nnimap-possibly-change-group group nil)
		(let ((imap-message-data (make-vector 2 0))
		      (exists (imap-folder-get 'EXISTS))
		      articles)
		  (if (eq 0 exists)
		      (with-current-buffer nntp-server-buffer
			    (insert (format "%s 0 1 y\n" group)))
		    (progn
		      (when (nnimap-ok-p (nnimap-send-command-wait
					  (concat "FETCH 1,* (UID)")))
			(imap-message-map (lambda (uid Uid) 
					    (push uid articles)) 'UID)
			(when articles
			  (with-current-buffer nntp-server-buffer
			    (insert (format "%s %d %d y\n" group
					    (apply 'max articles)
					    (apply 'min articles)))))))))))
	     ((eq nnimap-group-list-speed 'medium)
	      (gnus-message 10 "STATUS %s" group)
	      ;; Need to quote group if it has spaces!!!???
	      (when (nnimap-ok-p (nnimap-send-command-wait
				  (concat "STATUS " group 
					  " (UIDNEXT)")))
		(setq high (imap-folder-get 'UIDNEXT group))
		(with-current-buffer nntp-server-buffer
		  (insert (format "%s %d 1 y\n" group (- high 1))))))
	     (t ; (eq nnimap-group-list-speed 'fast)
	      (with-current-buffer nntp-server-buffer
		(insert (concat group " 0 1 y\n")))))))))

;;; Use the FAST way???  
;;; So the problem with the fast way is: what do we return if we don't want
;;; to say what the max messages are?  0 0 just makes him ask for message
;;; zero.  Gnus then thinks that there is
;;; 1 message is I return 1 1.  Boy this is annoying.
;;; Oh yea the new idea is 1 0.  Well see.
;;; With the fast way, after you subscribe, you can go a 'g' in the
;;; *Group* buffer to update the message counts (but only on the message
;;; you have subscribed to.
;;; Returns: GROUP HIGHEST LOWEST [ynmxj=]
(deffoo nnimap-request-list (&optional server)
  (when (nnimap-possibly-change-server server)
    (with-current-buffer nntp-server-buffer (erase-buffer))
    (with-current-buffer nnimap-server-buffer
      (setq imap-data-folder
            (make-vector imap-convenient-group-prime 0))
      (let* ((patterns (mapcar (lambda (p) (if (consp p) p (cons "\"\"" p)))
                               (if (or (atom nnimap-list-pattern)
                                       (atom (cdr nnimap-list-pattern)))
                                   (list nnimap-list-pattern)
                                 nnimap-list-pattern)))
             pattern)
        (while (setq pattern (pop patterns))
          (nnimap-ok-p (nnimap-send-command-wait
                        (concat nnimap-list-method " "
                                (car pattern) " "
                                (cdr pattern)))))
        (gnus-message 8 "Generating active list")
        (mapatoms nnimap-request-list-mapper imap-data-folder)))
    (setq nnimap-group-alist (nnmail-get-active))
    t))
  

;;; IMAP don't support posting
;(deffoo nnimap-request-post (&optional server)
;  nil)


(defun nnimap-possibly-change-server (server)
  (and (nnimap-open-server (nnoo-current-server 'nnimap))
       (nnoo-change-server 'nnimap server nil)))

(defun nnimap-possibly-change-group (group &optional server)
  (when (nnimap-possibly-change-server (or server imap-current-server))
    (with-current-buffer nnimap-server-buffer
      (unless (string= group imap-current-folder)
        (setq imap-message-data (make-vector 199 0)) ; itty bitty obarray???
        ;; Folder must be set during SELECT
        (setq imap-current-folder group)
        (unless (nnimap-ok-p (nnimap-send-command-wait
                              (concat (if (gnus-ephemeral-group-p group)
                                          "EXAMINE"
                                        "SELECT")
                                      " " group)))
          ;; Failed SELECT unselects the current group
          (setq imap-current-folder nil)))
      imap-current-folder)))

;(defun nnimap-update-flags (group info &optional server)
;  (when (nnimap-possibly-change-group group server)
;    (with-current-buffer nnimap-server-buffer
;      (when (nnimap-ok-p (nnimap-send-command-wait "FETCH 1:* (UID FLAGS)"))
;        (imap-message-map (lambda (uid flags)) 'FLAGS)))))

; nnimap-request-scan doesn't need to do anything. the delivery agent
; program is responsible for putting new message in the imap folders.
; compare the situation with nntp (nil function) and with all nnmail-
; backends (fetches mail from spools, POPing, saving in files on local
; storage etc).

;(deffoo nnimap-request-update-mark (group article mark)
;  mark)

(defun nnimap-update-alist-soft (key value alist)
  (if value 
      (cons (cons key value) (remassoc key alist))
    (remassoc key alist)))

;;; I really think this should update the active-info too???
(deffoo nnimap-request-update-info (group info &optional server)
  (when (nnimap-possibly-change-group group server)
    (with-current-buffer nnimap-server-buffer
      ;; extend info to have parameters (otherwise when we set them, 
      ;; they don't get set)
      (when (gnus-info-params info) 
	(gnus-info-set-method info nil) ;; something is setting this to server name???
	(gnus-info-set-params info nil t))
      
      ;; As long as the UIDVALIDITY is the same, UIDs
      ;; are vaild and persist (so save it)
      (gnus-group-set-parameter group 'uidvalidity
                                (imap-folder-get 'uidvalidity))
      ;; Replace list of read and marked articles with authoritative
      ;; data from server
      (gnus-info-set-read 
       info 
       (gnus-compress-sequence (nnimap-search "SEEN")))
      (if (gnus-info-read info)
	  (gnus-info-set-read info (list (gnus-info-read info)))) ; xxx dead ugly
      (gnus-info-set-marks 
       info 
       (nnimap-update-alist-soft 'tick (gnus-compress-sequence 
                                        (nnimap-search "FLAGGED"))
                                 (gnus-info-marks info)))
      (gnus-info-set-marks
       info 
       (nnimap-update-alist-soft 'reply (gnus-compress-sequence 
                                         (nnimap-search "ANSWERED"))
                                 (gnus-info-marks info))))
    info))
      
;;; Respond to articles with mail
(deffoo nnimap-request-type (group article)
  'mail)

;;; Notice that we don't actually delete anything, we just mark them
;;; deleted.  We need some kind of EXPUNGE command.  Also, when we
;;; expunge we delete all marked \Deleted articles.  So we either
;;; need to unmark,expunge,re-mark or come up with some new scheme.
;;; I guess force means delete, not force means expire.  We don't to
;;; any expiry currently.
;;; Also, we can search for articles older than some expiry time if
;;; we need to.  I'm not sure what to do.
(deffoo nnimap-request-expire-articles 
    (articles group &optional server force)
  (when force
    (when (nnimap-possibly-change-group group server)
      (with-current-buffer nnimap-server-buffer
;;;        (setq deleted (nnimap-search "DELETED"))
;;;        (nnimap-ok-p (nnimap-send-command-wait "EXPUNGE"))
;;;        (setq articles (gnus-sorted-complement articles deleted)))))
        (when (nnimap-ok-p (nnimap-send-command-wait 
                            (concat "UID STORE "
                                    (nnimap-range-to-string
                                     (gnus-compress-sequence articles))
                                    " +FLAGS (\\Deleted)")))
          (setq articles nil)))))
  ;; return articles not deleted
  articles)

;;; Maybe we should add to current group-alist???
(deffoo nnimap-request-create-group (group &optional server args)
  (when (nnimap-possibly-change-server server)
    (nnimap-ok-p (nnimap-send-command-wait
                  (concat "CREATE " group) 
                  nnimap-server-buffer))))

;;; Maybe we should remove from group-alist and unset if current group???
(deffoo nnimap-request-delete-group (group force &optional server)
  (when (nnimap-possibly-change-server server)
    (nnimap-ok-p (nnimap-send-command-wait
                  (concat "DELETE " group)
                  nnimap-server-buffer))))

;;; Maybe we should update the group-alist and current group???
(deffoo nnimap-request-rename-group (group new-name &optional server)
  (when (nnimap-possibly-change-server server)
    (nnimap-ok-p (nnimap-send-command-wait
                  (concat "RENAME " group " " new-name)
                  nnimap-server-buffer))))

;;; Maybe we should update the group-alist???
(deffoo nnimap-request-accept-article (group &optional server last)
  (and
   (nnimap-possibly-change-server server)
   ;; The article will be appended as UIDNEXT
   (nnimap-ok-p (nnimap-send-command-wait
                   (concat "STATUS " group " (UIDNEXT)")
                   nnimap-server-buffer))
   (nnimap-ok-p (nnimap-send-command-wait 
                   ;; Optional flags,date???
                   (list (format "APPEND %s " group) (current-buffer))
                   nnimap-server-buffer))
   (let ((high (imap-folder-get 'UIDNEXT group nnimap-server-buffer)))
     (when high
       (cons group high)))))

(provide 'nnimap)
