;;; imap.el --- IMAP library for emacs

;;; Copyright (C) 1998 Simon Josefsson
;;; Copyright (C) 1998 Jim Radford
;;; Copyright (C) 1997 John McClary Prevost

;;; This program is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 2 of the License, or
;;; (at your option) any later version.

;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.

;;; You should have received a copy of the GNU General Public License
;;; along with this program; if not, write to the Free Software
;;; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

;;; Commentary:

;;; This file provides a low level interface to an IMAP server.
;;; It provides functions to send commands and wait for responses.
;;; The results are parsed into lisp expressions and returned.
;;; IMAP untagged responses are stuffed in to obarrays based on
;;; the current group.  All the parameter variables become local
;;; to the IMAP process buffer.

;;; RFC1730 (IMAP4): done
;;; RFC1731 (Authentication mecanisms): currently only support for KERBEROS_V4
;;; RFC2060 (IMAP4rev1): done
;;; RFC???? (UNSELECT ext): done
;;; RFC2195 (CRAM-MD5 auth): done
;;; RFC2086 (ACL ext): done
;;; RFC2342 (NAMESPACE ext): done
;;; RFC2359 (UIDPLUS ext): done

;;; Todo:
;;; 
;;; o On expunge, remove messages from message-data. Note it doesn't
;;;   return UIDs.  Ouch.
;;; o Rename "folder" to "mailbox".
;;; 

(eval-when-compile (require 'cl))

(eval-and-compile
  (autoload 'open-ssl-stream "ssl")
  (unless (fboundp 'open-network-stream)
    (require 'tcp)))

;;; External variables

(defvar imap-default-port 143
  "*Default port number to be used for IMAP connections.  This should
probably be \"imap\", but a lot of machines lack the services entry.

This can be overrided by the server definition imap-port, and is the
prefered way of specifying this.")

(defvar imap-convenient-group-prime 2999
  "*A convenient prime which will be used to set the size of the group hash.
We have a lot of groups at CMU, so this should probably be adjusted down.")

(defvar imap-convenient-folder-prime 997
  "*A convenient prime which will be used to set the size of the folder
(message) hash.")

(defvar imap-open-stream nil
  "*The name of a function to use for opening an imap stream. Defaults on
nil to open a networked stream to the server.

Examples; imap-open-imtest-stream, imap-open-ssl-stream.

This can be overrided by the server definition imap-open-stram, and
this is the prefered way of specifying this.")

(defvar imap-auth-method nil
  "*The name of a function to use for loging on to the server. Defaults on
nil to plain text logins using the LOGIN command.

Examples; imap-authenticate-cram-md5.

This can be overried by the server definition imap-auth-method, and
this is the prefered way of specifying this.")

(defvar imap-eol "\r\n"
  "*The string sent to end a command.")

;; remove?
(defvar imap-default-name nil
  "*Your name, should you choose to accept it.")

(defvar imap-last-status nil
  "*Status returned by last IMAP command")

(defvar imap-timeout 60
  "*Timeout in seconds to wait for server response.")

(defvar imap-username nil
  "Username for server. ")

(defvar imap-password nil
  "Password for server.")

(defvar imap-cb-fetch-hook nil
  "Hook called when receiving a FETCH response. Called with article NUM,
FETCH and DATA response.")

;;; Internal variables

(defvar imap-authinfo nil
  "Buffer local variable which contains (user . password) for server.")

(defvar imap-process nil
  "The active process for the current IMAP buffer.")

(defvar imap-data-capability nil
  "Current server's capability list")

(defvar imap-data-namespace nil
  "Current server's namespace.")

(defvar imap-data-folder nil
  "Obarray which contains group information.")

(defvar imap-cb-finished-tags '()
  "Alist of tags which are completed but not yet handled.")

(defvar imap-message-data nil
  "Obarray which contains current message cache.")

(defvar imap-current-folder nil
  "Name of the current folder")

(defvar imap-current-message nil
  "Symbol of current message.")

(defvar imap-current-server nil
  "Name of current server machine.")

(defvar imap-tag-num 0
  "Number for tag increment.")

(defvar imap-tag-char ?A
  "Unique tag char per connection.")

(defvar imap-connection-number 0
  "Unique tag char per connection.")

(defvar imap-do-login t
  "Wheter imap-authenticate should try to log in or not.

This is normally only turned off by a `imap-open-stream' that does
it's own authentication.")

(defvar imap-cb-function-alist '((OK . imap-cb-response)
				 (NO . imap-cb-response)
				 (BAD . imap-cb-response)
				 (PREAUTH . imap-cb-response)
				 (BYE . imap-cb-bye)
				 (EXISTS . imap-cb-numbered)
				 (EXPUNGE . imap-cb-numbered)
				 (RECENT . imap-cb-numbered)
				 (CAPABILITY . imap-cb-capability)
				 (LIST . imap-cb-list)
				 (LSUB . imap-cb-list)
				 (FLAGS . imap-cb-flags)
				 (FETCH . imap-cb-fetch)
				 (SEARCH . imap-cb-search)
				 (STATUS . imap-cb-status)
				 (ACL . imap-cb-acl)
				 (NAMESPACE . imap-cb-namespace)
				 (default . imap-cb-default))
  "Alist of IMAP code to function callbacks.")

(defvar imap-cb-tag-alist '()
  "Alist of tags to callbacks for tagged responses.")

(defvar imap-locals '(imap-cb-finished-tags
                      imap-cb-tag-alist
                      imap-current-folder
                      imap-current-message
                      imap-current-server
                      imap-data-capability
		      imap-data-namespace
                      imap-data-folder
		      imap-open-stream
		      imap-auth-method
                      imap-do-login
                      imap-message-data
                      imap-default-name
		      imap-authinfo
                      imap-last-status
                      imap-process
                      imap-tag-num
                      imap-tag-char
                      imap-eol)
  "A list the variables that define an individual imap connection.
They are reset from their `default-value's .   You can pass values for
any of these to `imap-open-server'.")


;; If non nil these hold the name of a buffer to put debug into
(defvar imap-log   "*imap-log*")   ; imap session trace
(defvar imap-last  nil);"*imap-last*")  ; last line we attempted to parse
(defvar imap-debug nil);"*imap-debug*") ; random debug spew

(defsubst imap-disable-multibyte ()
  "Enable multibyte in the current buffer."
  (when (fboundp 'set-buffer-multibyte)
    (set-buffer-multibyte nil)))

(when imap-debug ; (untrace-all)
  (require 'trace)
  (buffer-disable-undo (get-buffer-create imap-debug))
  (imap-disable-multibyte)
  (mapc (lambda (f) (trace-function-background f imap-debug)) 
        '(imap-open-server
          imap-close-server
          imap-server-opened
	  imap-select-folder
	  imap-unselect-folder
	  imap-expunge-close-folder
          imap-send-command
          imap-send-command-wait
          imap-send-commands-wait
;	  imap-ok-p
	  imap-wait-for-tag
	  imap-capability-get
	  imap-namespace-get
	  imap-authinfo-get
          imap-folder-set
          imap-folder-get
          imap-folder-plist
	  imap-folder-reset
          imap-dispatch
	  imap-authenticate
	  imap-authenticate-login
	  imap-authenticate-cram-md5
	  imap-search
          imap-message-set
          imap-message-get
          imap-message-map
          imap-message-plist
	  imap-message-reset
          imap-cb-response
          imap-cb-bye
          imap-cb-numbered
          imap-cb-capability
	  imap-cb-namespace
          imap-cb-list
          imap-cb-flags
          imap-cb-fetch
          imap-cb-search
          imap-cb-status
          imap-cb-default)))

;;; Compatibility

(or (fboundp 'char-int)
    (fset 'char-int (symbol-function 'identity)))

(or (fboundp 'int-char)
    (fset 'int-char (symbol-function 'identity)))

(if (not (fboundp 'remassoc))
    (defun remassoc (key alist)
      "Delete by side effect any elements of LIST whose car is `equal' to KEY.
The modified LIST is returned.  If the first member of LIST has a car
that is `equal' to KEY, there is no way to remove it by side effect;
therefore, write `(setq foo (remassoc key foo))' to be sure of changing
the value of `foo'."
      (when alist
	  (if (equal key (caar alist))
	      (cdr alist)
	    (setcdr alist (remassoc key (cdr alist)))
	    alist))))

(if (not (fboundp 'save-current-buffer))
    (defmacro save-current-buffer (&rest body)
      "Save the current buffer; execute BODY; restore the current buffer.
Executes BODY just like `progn'."
      (` (let ((orig-buffer (current-buffer)))
	   (unwind-protect
	       (progn (,@ body))
	     (set-buffer orig-buffer))))))

(if (not (fboundp 'with-current-buffer))
    (defmacro with-current-buffer (buffer &rest body)
      "Execute the forms in BODY with BUFFER as the current buffer.
The value returned is the value of the last form in BODY.
See also `with-temp-buffer'."
      `(save-current-buffer
         (set-buffer ,buffer)
         ,@body)))

(if (not (fboundp 'destructive-plist-to-alist)) ;; From XEmacs subr.el 
    (defun destructive-plist-to-alist (plist)
      "Convert property list PLIST into the equivalent association-list form.
The alist is returned.  This converts from

\(a 1 b 2 c 3)

into

\((a . 1) (b . 2) (c . 3))

The original plist is destroyed in the process of constructing the alist.
See also `plist-to-alist'."
      (let ((head plist)
            next)
        (while plist
          ;; remember the next plist pair.
          (setq next (cddr plist))
          ;; make the cons holding the property value into the alist element.
          (setcdr (cdr plist) (cadr plist))
          (setcar (cdr plist) (car plist))
          ;; reattach into alist form.
          (setcar plist (cdr plist))
          (setcdr plist next)
          (setq plist next))
        head)))

;;; Interface functions

(defun imap-server-opened (&optional buffer)
  (with-current-buffer (or buffer (current-buffer))
    (and imap-process
         (member (process-status imap-process) '(open run)))))

(defun imap-close-server (&optional buffer autologout)
  "Logout if needed and close down the process.  Clean out buffer.
Ensure all `imap-locals' are local and reset them to their default
values such that the buffer will be suitable for opening a new server."
  ;; What is this for???
  (setq buffer (get-buffer (or buffer (current-buffer))))
  (when buffer
    (with-current-buffer buffer
      (mapc 'make-variable-buffer-local imap-locals) ; just in case
      (when imap-process
        (and (member (process-status imap-process) '(open run))
             (imap-send-command-wait "LOGOUT"))
        (delete-process imap-process))
      (mapc (lambda (local) (set local (default-value local))) imap-locals)
      (erase-buffer)
      t)))

(defun imap-current-server (&optional buffer)
  (with-current-buffer (or buffer (current-buffer)) 
    imap-current-server))

(defun imap-authenticate-login (server &optional buffer)
  "Login to server using the LOGIN command."
  (with-current-buffer (or buffer (current-buffer))
    (and (imap-authinfo-get server)
	 (imap-ok-p (imap-send-command-wait 
		       (concat "LOGIN \"" (car imap-authinfo) 
			       "\" \"" (cdr imap-authinfo) "\""))))))

(defun imap-authenticate-cram-md5 (server &optional buffer)
  "Login to server using the AUTH CRAM-MD5 method."
  (require 'rfc2104)
  (require 'md5)
  (with-current-buffer (or buffer (current-buffer))
    (and (imap-authinfo-get server)
	 (memq 'AUTH=CRAM-MD5 (imap-capability-get))
	 (imap-ok-p 
	  (imap-send-command-wait
	   (list 
	    "AUTHENTICATE CRAM-MD5"
	    (lambda (challenge)
	      (let* ((decoded (base64-decode challenge))
		     (hash (rfc2104-hash 'md5 64 16 (cdr imap-authinfo) decoded))
		     (response (concat (car imap-authinfo) " " hash))
		     (encoded (base64-encode response)))
		encoded))))))))

(defun imap-authenticate (server &optional buffer)
  (if (not imap-do-login)
      t
    (with-current-buffer (or buffer (current-buffer))
      (if imap-auth-method
	  (funcall imap-auth-method server buffer)
	(imap-authenticate-login server buffer)))))

(defun imap-open-server (server &optional port buffer local-defs)
  (with-current-buffer (get-buffer-create (or buffer (current-buffer)))
    (buffer-disable-undo)
    (imap-disable-multibyte)
    (imap-close-server) ; makes vars local, sets them to their defaults, erases
    (mapc (lambda (ld) (set (car ld) (cdr ld))) local-defs)
    (when (setq imap-process 
                (imap-open-stream "imap" (current-buffer) 
                                  server (or port imap-default-port)))
      (set-marker (process-mark imap-process) (point-min))
      (set-process-filter imap-process 'imap-arrival-filter)
      (setq imap-current-server server)
      ;; Give each connection a more or less unique letter just so the log
      ;; is easy to read
      (setq imap-tag-char (int-char (+ (char-int ?A) 
				       (% imap-connection-number 26))))
      (setq imap-connection-number (1+ imap-connection-number))
      (setq imap-data-folder (make-vector imap-convenient-group-prime 0))
      (current-buffer))))

;; If there is a need for sending commands without a callback, then
;; have `imap-send-command-wait'ing commands pass 
;; `imap-cb-tag-default' itself.  Maybe `imap-wait-for-tag' should
;; be internal.  It currently doesn't accept a buffer (nor set one).
;; I guess until then this is an internal command also.  Use
;; `imap-send-command-wait' instead.

(defun imap-send-command (command &optional buffer callback)
  "Send a COMMAND to the server for BUFFER returning the command's TAG.  If
BUFFER is omitted or is nil the current buffer is used.  You can
then call `imap-wait-for-tag'. If CALLBACK is provided, then you cannot
call `imap-wait-for-tag'.

COMMAND may be a list of strings, buffers and/or functions which should
be `concat'ed together.  The buffers are sent as IMAP string literals.
The functions accept one argument and are called with a server challenge and
should return the client response or \"*\" to give up."
  (with-current-buffer (or buffer (current-buffer)) 
    (setq imap-last-status nil) ; Yuck
    (let* ((tag (format "%c%d" imap-tag-char (setq imap-tag-num (1+ imap-tag-num))))
           (commands (append (list tag " ") 
                            (if (listp command) command (list command))
                            (list imap-eol)))
           (strings))
      (setq imap-cb-tag-alist ;; removed by `imap-dispatch'
            (cons (cons tag (or callback 'imap-cb-tag-default))
                  imap-cb-tag-alist))
      (while commands
        (while (stringp (car commands)) ; This should be easier
          (push (car commands) strings) ; How to append without reversing???
          (pop commands))
        (when commands ; buffer or function
          (cond
           ((bufferp (car commands))
            (push (format "{%d}%s" (with-current-buffer (car commands)
                                     (buffer-size))
                          imap-eol)
                  strings))
           ((functionp (car commands))
            (push imap-eol strings)))
          (setq imap-cb-tag-alist ;; removed by `imap-dispatch'
                (cons (cons "+" 'imap-cb-tag-default) imap-cb-tag-alist)))
        (setq strings (apply 'concat (nreverse strings)))
        (and imap-log (with-current-buffer (get-buffer-create imap-log)
                        (buffer-disable-undo)
			(imap-disable-multibyte)
                        (goto-char (point-max))
                        (insert strings)))
        (process-send-string nil strings)
        (setq strings nil)
        (when commands ; buffer or function
          ;; Waiting for "+" and bail out if we get tag.
          (let ((data (imap-wait-for-tag "+" tag)))
            (if (not data)
                (setq commands nil) ; Don't bother sending the rest
              (cond 
               ((bufferp (car commands)) ; buffer
                (and imap-log (with-current-buffer (get-buffer-create imap-log)
                                (goto-char (point-max))
                                (insert-buffer-substring (car commands))))
                (let ((process imap-process))
                  (with-current-buffer (car commands)
                    (process-send-region process (point-min) (point-max))))
                (pop commands))
               ((functionp (car commands)) ; function
                ;; Assume function comes at the end.
                ;; Send result of function call by prepending it to
                ;; the list of command strings.
                ;; The function gets removed only after the end
                ;; of the exchange (by an `imap-wait-for-tag' bailout).
                (setq commands 
                      (cons (funcall (car commands) (car data)) 
			    commands))))))))
      tag)))

(defun imap-send-command-wait (command &optional buffer)
  "Send a COMMAND to the server for BUFFER with a new TAG, and wait for
the command to complete on the IMAP server before returning.  If buffer is
omitted, the current buffer is used.

COMMAND may be a list of strings and buffers which should
be `concat'ed together.  The buffers are sent as IMAP string literals."
  (interactive "sCommand: ")
  (with-current-buffer (or buffer (current-buffer))
    (imap-wait-for-tag (imap-send-command command))))

(defun imap-send-commands-wait (command-list &optional buffer)
  "Send a list of commands and wait for results.  Results are returned
in order.  See `imap-send-command-wait'."
  (with-current-buffer (or buffer (current-buffer))
    (mapcar 'imap-wait-for-tag 
            (mapcar 'imap-send-command command-list))))

(defun imap-ok-p (status)
  (when (and status
	     (eq 'OK (car status)))
    (setq imap-last-status nil)
    t))

(defun imap-search (predicate &optional buffer)
  (with-current-buffer (or buffer (current-buffer))
    (imap-folder-set 'search nil)
    (when (imap-ok-p (imap-send-command-wait (concat "UID SEARCH " predicate)))
      (imap-folder-get 'search))))

(defun imap-store-flags-set (articles flags &optional buffer silent)
  (when (and articles flags)
    (with-current-buffer (or buffer (current-buffer))
      (imap-ok-p (imap-send-command-wait
		  (concat "UID STORE " articles
			  " FLAGS" (if silent ".SILENT") " (" flags ")"))))))

(defun imap-store-flags-del (articles flags &optional buffer silent)
  (when (and articles flags)
    (with-current-buffer (or buffer (current-buffer))
      (imap-ok-p (imap-send-command-wait
		  (concat "UID STORE " articles
			  " -FLAGS" (if silent ".SILENT") " (" flags ")"))))))

(defun imap-store-flags-add (articles flags &optional buffer silent)
  (when (and articles flags)
    (with-current-buffer (or buffer (current-buffer))
      (imap-ok-p (imap-send-command-wait
		  (concat "UID STORE " articles
			  " +FLAGS" (if silent ".SILENT") " (" flags ")"))))))

(defun imap-select-folder (folder &optional buffer examine)
  (with-current-buffer (or buffer (current-buffer))
    ;; xxx: check SELECT/EXAMINE status!  This is BAD.
    (unless (string= folder imap-current-folder)   
      (setq imap-current-folder folder)
      (if (imap-ok-p (imap-send-command-wait
		      (concat (if examine "EXAMINE" "SELECT") " " folder)))
	  (setq imap-message-data (make-vector imap-convenient-folder-prime 0))
	;; Failed SELECT unselects the current group
	(setq imap-current-folder nil
	      imap-message-data nil)))
    imap-current-folder))

(defun imap-unselect-folder (&optional group buffer)
  "Close current folder in BUFFER, without expunging articles."
  (with-current-buffer (or buffer (current-buffer))
    (when (or (and (memq 'UNSELECT (imap-capability-get))
		   (imap-ok-p (imap-send-command-wait "UNSELECT")))
	      (and (imap-ok-p 
		    (imap-send-command-wait (concat "EXAMINE " 
						    (or group
							imap-current-folder))))
		   (imap-ok-p (imap-send-command-wait "CLOSE"))))
      (setq imap-current-folder nil
	    imap-message-data nil)
      t)))

(defun imap-expunge-close-folder (&optional buffer)
  "Expunge articles and close current folder in BUFFER."
  (with-current-buffer (or buffer (current-buffer))
    (when (imap-ok-p (imap-send-command-wait "CLOSE"))
      (setq imap-current-folder nil
	    imap-message-data nil)
      t)))

(defun imap-folder-lsub (&optional reference buffer)
  "Return a list of strings of subscribed mailboxes on server in
BUFFER. REFERENCE is the implementation-specific string that has to be
passed to LSUB."
  (with-current-buffer (or buffer (current-buffer))
    (imap-folder-reset)
    (when (imap-ok-p (imap-send-command-wait 
		      (concat "LSUB \"" reference "\" \"*\"")))
      (imap-folder-map 'identity))))

(defun imap-folder-list (&optional root have-delimiter reference buffer)
  "List all mailboxes that starts with ROOT in BUFFER. If
HAVE-DELIMITER is non-nil, a hierarchy delimiter is not added to
ROOT. REFERENCE is the implementation-specific string that has to be
passed to LIST."
  (with-current-buffer (or buffer (current-buffer))
    (imap-folder-reset)
    ;; Find hierarchy separator
    (unless have-delimiter
      (imap-send-command-wait (concat "LIST \"" reference "\" \"" root "\"")))
    (when (imap-ok-p 
	   (imap-send-command-wait 
	    (concat "LIST \"" reference "\" \"" root
		    (when (and (not have-delimiter) root)
		      (imap-folder-get 'delimiter root))
		    "%\"")))
      (imap-folder-map 'identity))))

(defun imap-folder-subscribe (mailbox &optional buffer)
  "Send the SUBSCRIBE command on the mailbox to server in
BUFFER. Returns non-nil if successful."
  (with-current-buffer (or buffer (current-buffer))
    (imap-ok-p (imap-send-command-wait (concat "SUBSCRIBE " mailbox)))))

(defun imap-folder-unsubscribe (mailbox &optional buffer)
  "Send the SUBSCRIBE command on the mailbox to server in
BUFFER. Returns non-nil if successful."
  (with-current-buffer (or buffer (current-buffer))
    (imap-ok-p (imap-send-command-wait (concat "UNSUBSCRIBE " mailbox)))))

;;; Variable setters and getters

(defun imap-capability-get (&optional buffer)
  "Return a list of identifiers which the server support."
  (with-current-buffer (or buffer (current-buffer))
    (unless imap-data-capability
      (unless (imap-send-command-wait "CAPABILITY")
	(setq imap-data-capability '(IMAP2))))
    imap-data-capability))

(defun imap-namespace-get (&optional buffer)
  "Return server's namespace."
  (with-current-buffer (or buffer (current-buffer))
    (unless imap-data-namespace
      (when (memq 'NAMESPACE (imap-capability-get))
	(imap-send-command-wait "NAMESPACE")))
    imap-data-namespace))

(defun imap-folder-plist (&optional folder buffer)
  "Set PROP to VALUE for FOLDER in BUFFER."
  (with-current-buffer (or buffer (current-buffer))
    (object-plist (intern (or folder
                              imap-current-folder)
                          imap-data-folder))))

(defun imap-folder-set (prop value &optional folder buffer)
  "Set PROP to VALUE for FOLDER in BUFFER."
  (with-current-buffer (or buffer (current-buffer))
    (put (intern (or folder
                     imap-current-folder)
                 imap-data-folder) prop value)))

(defun imap-folder-get (prop &optional folder buffer)
  "Get PROP for FOLDER or the current folder in BUFFER"
  (with-current-buffer (or buffer (current-buffer))
    (get (intern (or folder
                     imap-current-folder) imap-data-folder) prop)))

(defun imap-folder-map (func &optional buffer)
  "Call (func FOLDER) for each folder in `imap-data-folder', returning
a sequence."
  (with-current-buffer (or buffer (current-buffer))
    (let (result)
      (mapatoms 
       (lambda (s)
	 (push (funcall func (symbol-name s)) result))
       imap-data-folder)
      result)))

(defun imap-folder-reset (&optional buffer)
  (with-current-buffer (or buffer (current-buffer)) 
    (setq imap-data-folder (make-vector imap-convenient-group-prime 0))))

;;; Internal functions

(defun imap-read-passwd (prompt &rest args)
  "Read a password using PROMPT.
If ARGS, PROMPT is used as an argument to `format'."
  (let ((prompt (if args 
		    (apply 'format prompt args)
		  prompt)))
    (funcall (if (load "passwd" t)
		 'read-passwd
	       (unless (fboundp 'ange-ftp-read-passwd)
		 (autoload 'ange-ftp-read-passwd "ange-ftp"))
	       'ange-ftp-read-passwd) prompt)))

(defun imap-authinfo-get (server &optional buffer)
  "Get user authentication information. Uses imap-username and/or
imap-password. Asks the user if necessery. If successful, sets 
imap-authinfo to (username . password)."
  (with-current-buffer (or buffer (current-buffer))
    (let (user passwd)
      (setq user (or imap-username
		     (read-from-minibuffer (concat "IMAP Name for " server 
						   ": ")
					   imap-default-name)))
      (setq passwd (or imap-password
		       (imap-read-passwd (concat "IMAP Password for " user "@"
						 server ": "))))
      (if (and user passwd)
	  (progn
	    (setq imap-authinfo (cons user passwd))
	    t)
	(setq imap-authinfo nil)))))

(defun imap-open-stream (name buffer host &optional port)
  (let ((coding-system-for-read 'binary)
	(coding-system-for-write 'binary))
    (if imap-open-stream
	(funcall imap-open-stream name buffer host port)
      (imap-open-network-stream name buffer host port))))

(defun imap-open-network-stream (name buffer host &optional port)
  (open-network-stream name buffer host port))

(defun imap-open-ssl-stream (name buffer host &optional port)
  (let ((ssl-program-arguments '("-connect" (concat host ":" service)))
	(proc (open-ssl-stream name buffer host port)))
    (save-excursion
      (set-buffer buffer)
      (goto-char (point-min))
      (while (not (re-search-forward "^\r*\* OK" nil t))
	(accept-process-output proc imap-timeout)
	(goto-char (point-min)))
      (beginning-of-line)
      (delete-region (point-min) (point))
      proc)))

(defun imap-open-imtest-stream (name buffer host &optional port)
  (let ((process (start-process name (or buffer (current-buffer))
                                "imtest" "-kp" host 
				(number-to-string (or port 
						      imap-default-port)))))
    (with-current-buffer (process-buffer process)
      (setq imap-eol "\n")
      (setq imap-do-login nil) ;; don't login even if kerberos auth fails
      (when process
	(message "Opening Kerberized IMAP connection...")
	;; Result of authentication is a string: __Full privacy protection__
	(while (and (member (process-status imap-process) '(open run))
		    (not (or
			  ;; if everything is ok, this will match...
			  (re-search-backward "__\\(.*\\)__\n" nil t)
			  ;; ...errors will match this
			  (re-search-backward 
			   "\\. \\([^OA][^KU][^T][^H].*\\)\r\n" nil t))))
	  (accept-process-output process 1)
	  (sit-for 1)) ; Yes, this is an oo loop, allow for C-g
	(let ((response (match-string 1)))
	  (erase-buffer)
	  (message "Kerberized IMAP connection: %s" response)
	  ;; If the __string__ contains "failed" authentication failed
	  ;; (imtest will bug out if you try to login the usual way, so
	  ;; close connection with an error)
	  (when (string-match "failed\\|NO\\|BAD" response)
	    (mapc 'make-variable-buffer-local imap-locals) ; just in case
	    ;; XXX logout here (can't use send-command since we don't
	    ;; have the server opened..)
	    (delete-process imap-process)
	    (mapc (lambda (local) (set local (default-value local))) imap-locals)
	    (erase-buffer)
	    (error "imtest: %s" response))))
      process)))

(defun imap-arrival-filter (proc string)
  "Process filter for imap process.  Stow the string, then call the routines
to try to handle any input.  We need this because we're not guaranteed to
receive the whole thing at once."
  (with-current-buffer (process-buffer proc)
    (goto-char (point-max))
    (insert string)
    ;; Keep a log of server tranactions in `imap-log'
    (when imap-log
      (with-current-buffer (get-buffer-create imap-log)
        (goto-char (point-max))
        (insert string)))
    (let (end)
      ;; Find compete server line
      (while (setq end (imap-find-next-line))
        (save-restriction
          ;; Restrict to it
          (narrow-to-region (point-min) end)
          ;; Parse and then dispatch
          (unwind-protect (apply 'imap-dispatch (imap-parse-line))
            ;; Delete it ensuring that the parser doesn't get out of
            ;; sync on errors by leaving half parsed stuff around
            (delete-region (point-min) (point-max))))))))
        
;;; Callback dispatching

(defun imap-get-cb (name)
  "Get the callback associated with the given result type, or use the default."
  (cdr (or (assoc name imap-cb-function-alist)
           (assoc 'default imap-cb-function-alist))))

(defun imap-dispatch (tag response &rest data)
  "Take the parsed IMAP sludge and figure out who can deal with it.  When
the guy is found, send it to him to process.";     Tag Response Data
  (unless (eq tag '+)
    (apply (imap-get-cb (if (numberp response) ; Ex:  *    3      EXISTS
                            (car data)         ; Ex: A25   OK     LOGIN
                          response))           ; Ex:  *  SEARCH   1 2 3
           (cons response data))) ; allow callee to decide the number of args
  (if (not (eq tag '*))
      (let ((fn (assoc (symbol-name tag) imap-cb-tag-alist)))
         (setq imap-last-status data) ; Yuck
         (if (not fn)
             (error "No callback for %s" (symbol-name tag))
           (funcall (cdr fn) tag response data)
           ;; (apply (cdr fn) (list tag response data))
           (setq imap-cb-tag-alist 
                 (remassoc (symbol-name tag) imap-cb-tag-alist))))))

;;; IMAP Notes:
;;;  o RESPONSE is one of (OK NO BAD PREAUTH BYE)
;;;  o TAG is a unique identifier for COMMAND
;;;  o Response to TAG/COMMAND has matching TAG/COMMAND
;;;  o TEXT is an unquoted human readable string
;;;  o DATA is parenthesized list of lists of space separated literals or
;;;     quoted strings
;;;  o quoted strings are either {LENGTH}\r\nUNQUOTED_DATA
;;;    or a "" string with with '\' quoting.
;;;    We assume that if something need quoting the server would use {} not ""
;;;  o literals have no spaces and can contain '\'s.
;;;  o CODE is one of (TRYCREATE READ-ONLY ALERT ...)

;;; Grammar (condensed)
;;;   IMAP command 
;;;     TAG COMMAND DATA*
;;;   IMAP responses
;;;     + TEXT
;;;     TAG RESPONSE [CODE DATA?] COMMAND TEXT
;;;     * RESPONSE [CODE DATA?] TEXT
;;;     * WORD DATA*
;;;     * NUMBER WORD



;;; Basic tokenizing and parsing
;;; Well--not really.  Listen up, this is sort of gross.  What we do is
;;; scan across the input, munching as we go.  We convert anything
;;; unfriendly to something friendlier (usually \ it), then use (read)
;;; to gobble up the input.
;;;  I told you it was gross.

(defun imap-find-next-line ()
  "Find the next available input into the buffer.  This is similar to the
actual scanning code, except that it doesn't munch or read anything.  All
it does is verify that there's a complete response in the buffer, and return
the position of the end of the response.  If there is no complete response,
this returns nil."
  (goto-char (point-min))
  (let ((finished nil)
        (jump-amount nil)
        (return-val nil))
    ;; We're looking for three things--
    ;;  {num} is a string literal--skip it.
    ;;  "..." is a string--skip it, too.
    ;;  CRLF is what we see (not in a string) when we're done.
    (while (and (not finished)
                (re-search-forward
                 "\"[^\"]*\"\\|\r\n\\|{\\([^}]+\\)}" nil t))
      (and (match-string 1)
           (setq jump-amount (string-to-number (match-string 1))))
      (cond ((eq (preceding-char) ?\n)
             (setq finished t)
             (setq return-val (point)))
            ((eq (preceding-char) ?\})
             (if (< (point-max)
                    (+ (point) jump-amount 2))
                 (setq finished t)
               (goto-char (+ (point) jump-amount 2))))))
    return-val))

(defun imap-parse-line () 

  ;; Parse one server response.  We do translations to buffer so we
  ;; can use `read'.  We quote the UIDVALIDITY number to avoid 28-bit
  ;; integer limitations.  Quotes [] into {} so that things like
  ;; BODY[HEADER] are one symbol, backslashify things, turn CRLF into
  ;; LF.  Call this at the start of the block to suck in, narrowed to
  ;; the entirety of the block (you should know the end of it...).

  ;; The only place we could have a problem is in the arbitrary TEXT
  ;; following a status response.  So we quote that first, and be
  ;; sure to remove all " and \'s that we can't handle.

  (let ((leave-brackets 0))
    (goto-char (point-min))
    (insert "(")
    (when (re-search-forward
           "\\=\\(\\(\\+\\)\\|[^ ]+ +\\(OK\\|NO\\|BAD\\|PREAUTH\\|BYE\\)\\( +\\[.*?\\]\\)?\\) *"
           nil t)
      (unless (or (match-string 2) (match-string 4))
        (insert "[] "))
      (setq leave-brackets (if (match-string 3) (point-marker) (point-min-marker)))
      (save-restriction
        (narrow-to-region (point) (- (point-max) 2))
        (insert "\"")
        (while (re-search-forward "\\\\\\|\"" nil 00)
          (replace-match "")) ; xxx
        (insert "\""))
      (goto-char (point-min)))
    (goto-char (- (point-max) 2))
    (insert ")")
    (goto-char (point-min))
    (let ((finished nil)
          (jump-amount nil))
      (while (and (not finished)
                  (re-search-forward
                   (mapconcat 'identity
                              '("\"[^\"]*\"" ; quoted strings
                                "\\[\\|\\]" ; [] characters
                                "\\(UIDVALIDITY\\|COPYUID\\|APPENDUID\\) \\([0123456789]+\\)"
                                "\\."   ; . characters
                                "\\#"   ; # characters
                                "\\\\"  ; \ characters
                                "\r\n"  ; CRLF
                                "{\\([^}]+\\)}") ; string literals
                              "\\|") nil t)) ; regexp or
        (and (match-string 3)
             (setq jump-amount (string-to-number (match-string 3))))
        (let ((pc (preceding-char)))
          (cond ((eq pc ?\n)
                 (setq finished t))
                ((member pc '(?\\ ?. ?#))
                 (backward-char)
                 (insert "\\")
                 (forward-char))
                ((and (eq pc ?\[) (> (point) leave-brackets))
                 (replace-match "{"))
                ((and (eq pc ?\]) (> (point) leave-brackets))
                 (replace-match "}"))
                ((and (>= pc ?0)
                      (<= pc ?9))
                 ;; replace numbers too big for emacs with strings
                 (replace-match "\\1 \"\\2\""))
                ((eq pc ?\})
                 (delete-region (match-beginning 0) (match-end 0))
                 (delete-char 2)
                 (save-restriction;; save-res is good; we change buffer size.
                   (narrow-to-region (point) (+ (point) jump-amount))
                   (insert "\"")
                   (while (re-search-forward "\\\\\\|\"" nil 00)
                     (replace-match "\\\\\\&"))
                   (goto-char (point-min))
                   (while (search-forward "\r\n" nil 00)
                     (replace-match "\n" nil t))
                   (insert "\"")))))))
    (goto-char (point-min))

    (and imap-last
         (let ((buffer (current-buffer)))
           (with-current-buffer (get-buffer-create imap-last)
             (buffer-disable-undo)
	     (imap-disable-multibyte)
             (erase-buffer)
             (insert-buffer-substring buffer))))
      
    (read (current-buffer))))
  
(defun imap-cb-default (&rest a)
  (error "Default Callback Called %s" a))

(defun imap-cb-list (code taglist delim name-symbol-or-string)
  (let ((name (if (symbolp name-symbol-or-string)
                  (symbol-name name-symbol-or-string)
                name-symbol-or-string)))
    (imap-folder-set 'flags (mapcar 'symbol-name taglist) name)
    (imap-folder-set 'delimiter delim name)
    (when (eq code 'LSUB) 
      (imap-folder-set 'subbed t name))))

(defun imap-cb-flags (code flags)
  (imap-folder-set 'list-flags (mapcar 'symbol-name flags)))

(defun imap-message-to-string (message)
  (if (numberp message)
      (format "\\%s" message)
    message))

(defun imap-message-plist (id &optional buffer)
  "Set PROP to VALUE for message ID in buffer."
  (with-current-buffer (or buffer (current-buffer))
    (object-plist (intern (imap-message-plist id) imap-message-data))))

(defun imap-message-map (func prop &optional buffer)
  "Call (func UID VALUE) for each message in `imap-message-data'.
VALUE is the value of PROP for each message."
  (with-current-buffer (or buffer (current-buffer))
    (mapatoms 
     (lambda (s)
       (funcall func (get s 'UID) (get s prop)))
     imap-message-data)))

(defun imap-message-set (id prop value &optional buffer)
  "Set PROP to VALUE for message ID in buffer."
  (with-current-buffer (or buffer (current-buffer))
    (put (intern (imap-message-to-string id) imap-message-data) prop value)))

(defun imap-message-get (id prop &optional buffer)
  "Get PROP for message ID in BUFFER."
  (with-current-buffer (or buffer (current-buffer))  
   (get (intern (imap-message-to-string id) imap-message-data) prop)))

(defun imap-message-reset (&optional buffer)
  (with-current-buffer (or buffer (current-buffer)) 
    (setq imap-message-data (make-vector imap-convenient-folder-prime 0))))

; Fetches MUST include (UID) otherwise we can't store the results.
; NUM is always the logical message number not the UID.  We ignore it.
(defun imap-cb-fetch (num fetch data)
  "Set `imap-current-message', Set all of the prop/value pairs
in `imap-message-data'."
  (when imap-cb-fetch-hook
    (funcall imap-cb-fetch-hook num fetch data))
;  (check-valid-plist data) ; Remove me when you have confidence???
  (setq data (destructive-plist-to-alist data))
  ;; All fetches should have used UID FETCH so UID should exist.
  ;; UID STORE will not have a UID in the data, in this case the UID is num
  (setq imap-current-message (or (cdr (assoc 'UID data)) num))
  (mapc (lambda (c) (imap-message-set imap-current-message (car c) (cdr c)))
        data))

(defun imap-cb-search (search &rest found)
  ;; For some reason found doesn't have parens so we need the &rest
  (imap-folder-set 'search found))

(defun imap-cb-capability (code &rest capabilities)
  (setq imap-data-capability capabilities))

(defun imap-cb-acl (code group &rest acls)
  (imap-folder-set 'acl acls (symbol-name group)))

(defun imap-cb-namespace (code &rest namespace)
  (setq imap-data-namespace namespace))

(defun imap-cb-status (code folder statuses)
;  (check-valid-plist statuses)
  (mapc (lambda (c) (imap-folder-set (car c) (cdr c) (symbol-name folder)))
        (destructive-plist-to-alist statuses)))

(defun imap-cb-numbered (number code) ; These are just numbers not UIDs
  (cond ((eq 'EXISTS code)
	 (imap-folder-set 'EXISTS number))
	((eq 'RECENT code)
	 (imap-folder-set 'RECENT number))))

(defun imap-cb-bye (bye nothing string)
  "Called on BYE.   This is most likely an autologout.  Clean up."
  ;; Do something here???
  )

(defun imap-cb-response (response code string)
  (setq code (append code '()))
  (let ((var (car code))
        (value (cadr code)))
    ;; missing NEWNAME, PARSE
    (when var
      (cond ((equal var 'ALERT)
             (y-or-n-p (concat "IMAP: ALERT! " string)))
            ((equal var 'APPENDUID) ;; xxx: set in mailbox we're appending to
             (imap-folder-set 'appenduid (cons value (caddr code))))
            ((equal var 'PERMANENTFLAGS)
             (imap-folder-set 'permanentflags (mapcar 'symbol-name value)))
            ((equal var 'READ-ONLY)
             (imap-folder-set 'writable nil))
            ((equal var 'READ-WRITE)
             (imap-folder-set 'writable t))
	    ((equal var 'TRYCREATE)
	     (message "IMAP: %s %s" code string))
	    ((equal var 'UIDNEXT)
	     (imap-folder-set 'uidnext value))
            ((equal var 'UNSEEN)
             (imap-folder-set 'unseen value))
            ((equal var 'UIDVALIDITY) ; We should wipe cache clean here only???
             (imap-folder-set 'uidvalidity value))
	    ((equal var 'COPYUID)
	     (imap-folder-set 'copyuid value))
            (t (error "IMAP: Unknown response code: %s %s" code string))))))

(defun imap-cb-tag-default (tag response data)
  "This will put args on `imap-cb-finished-tags' for `imap-wait-for-tag'."
  (let ((stats (list response data)))
    (setq imap-cb-finished-tags (cons (cons (symbol-name tag) stats)
				      imap-cb-finished-tags))))

(defun imap-wait-for-tag (tag &optional bail)
  "Wait for TAG to complete by calling `imap-cb-tag-default' and return
its args.  Stop waiting if BAIL is seen"
  (let (data bailed)
    (while (when (not (or (setq data (assoc tag imap-cb-finished-tags))
                          (setq bailed (assoc bail imap-cb-finished-tags))))
             ;; timeout after imap-timeout seconds with no data
             (accept-process-output imap-process imap-timeout)))
    (unless (or data bailed)
      (message "IMAP: Timed out waiting for %s" tag))
    (when (or data bailed)
      (setq imap-cb-finished-tags (remassoc tag imap-cb-finished-tags))
      (cdr data))))

(provide 'imap)
