;;; nnimap.el --- IMAP backend for Gnus

;;; Copyright (C) 1998 Simon Josefsson <jas@pdc.kth.se>
;;; Copyright (C) 1998 Jim Radford <radford@robby.caltech.edu>
;;; Copyright (C) 1997 John McClary Prevost <visigoth@cs.cmu.edu>

;;; This program is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 2 of the License, or
;;; (at your option) any later version.

;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.

;;; You should have received a copy of the GNU General Public License
;;; along with this program; if not, write to the Free Software
;;; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

;;; Commentary:

;;; This file provides IMAP support for the Emacs news/mailreader Gnus.
;;; To enable this backend you put the following in your .gnus:

;;;  (require  'nnimap)
;;;  (setq gnus-secondary-select-methods 
;;;      '((nnimap 
;;;          "yoyo"
;;;          (nnimap-address "robby.caltech.edu"))))

;;; And inside Gnus, list all available groups with A A (search for 
;;; 'nnimap') and subscribe to the mailboxes you are interested in with U. 
;;; If you know the name of the mailbox, you can also use 'U'
;;; (gnus-group-unsubscribe-group) to subscribe to it (no mailbox
;;; completion here, sorry).

;;; Note that Gnus can't deal with groups starting with '.' (a dot),
;;; this means that you will have to use ~/iMail/ instead of ./iMail
;;; (or similair).

;;; Note also that you shouldn't subscribe to goups on your nnimap
;;; server from the server buffer if the server is your primary server
;;; i.e. in `gnus-select-method'.  I think this is a bug in Gnus.
;;; Yep, I just checked.  It is fixed in at least 5.6.27.  It was
;;; still broken in 5.5.

;;; Todo (roughly in order of priority):

;;;   o What about Gnus's article editing, can we support it?
;;;   o Move common IMAP commands to functions in imap.el.
;;;   o Verify that we don't use IMAP4rev1 specific things (RFC2060 App B)
;;;   o dont uid fetch 1,* in nnimap-retrive-groups (slow)
;;;   o Split up big fetches (1,* header especially) in smaller chunks
;;;   o use \Draft to support the draft group??
;;;   o What do I do with gnus-newsgroup-*?
;;;   o Figure out when update-info is called and why and if we need to
;;;     call it more often and from where.
;;;   o Add update-info calls? To request-group, close-group, retrieve-groups
;;;   o Tell Gnus about new groups (how can we tell?)
;;;   o Add asynchronous support with two server connections (use BODY.PEEK)
;;;   o Bulletin board stuff.
;;;   o Respooling (fix Gnus?)
;;;   o Subscribe to newly created mailboxes (how? nnimap-request-newgroups?)
;;;   o Add support for the following: (if applicable)

;;;       request-list-newsgroups, request-regenerate
;;;       list-active-group,
;;;       request-associate-buffer, request-restore-buffer,

;;;   o When UIDVALIDITY changed, I should reconstruct everything gnus knows
;;;     about the group (possible?) (fixed?)
;;;   o split to other backends, different split rules for different
;;;     servers/inboxes
;;;   o Fix Gnus to handle SPC and leading '.' in group names
;;;   o Fix the flag situation when using the Gnus Agent
;;;   o Support RFC2221 (Login referrals)
;;;   o IMAP2BIS compatibility? (RFC2061)
;;;   o Debug imtest, it dumps with "Time is out of bounds" sometimes
;;;   o ACAP stuff (perhaps a different project, would be nice to ACAPify 
;;;     .newsrc.eld)
;;;   o MIME

(require 'imap)

(require 'nnoo)
(require 'nnheader)
(require 'nnmail)
(require 'gnus)
(require 'gnus-range)

(eval-when-compile (require 'cl))

(gnus-declare-backend "nnimap" 'mail 'address 'prompt-address 
		      'physical-address 'respool)  ;; respool??

(nnoo-declare nnimap) ; we derive from no one

(defconst nnimap-version "nnimap 0.81")

;; Various server variables.

(defvoo nnimap-directory message-directory
  "Data directory for the nnimap backend.")

(defvoo nnimap-active-file
  (concat (file-name-as-directory nnimap-directory) "active")
  "Mail active file for the nnimap backend.")

(defvoo nnimap-list-pattern "*" 
"*PATTERN or list of PATTERNS use to limit available groups.  

A pattern is either GROUP or (REFERENCE . GROUP).

GROUP is a string.  See the available wildcard characters below.

The meaning of REFERENCE is server-specific, so it's
expected that you (the user) can figure out the appropriate setting.
On the Cyrus server, this is irrelevant.  On the UWash server, this
gets joined together with GROUP.  If it is not specified the
default is an empty string.

Example:
 '(\"INBOX\" \"Mail/*\" \"alt.sex.*\" (\"~friend/Mail/\" . \"list/*\"))

Also note that currently groups that start with a '.' cause Gnus to
choke, so instead of using something like \"*\" which might match
\".mailboxlist\" you could use \"~/*\" which would match
\"~/.mailboxlist\" and not cause problems.

The two wildcards are * and %.  * means match anything, much like in
shell globbing in Unix.  * does match hierarchy delimiters (. or /, in
the usual case.)  % is the same as *, but does not match the hierarchy
delimiter.")

(defvoo nnimap-list-method "LIST" ; "LSUB"
  "*Function called on IMAP server to list groups.  One of \"LSUB\" or
\"LIST\". LSUB means only retrieve groups marked on the server as
subscribed.  LIST means every matching group should be retrieved.")

(defvoo nnimap-address nil
  "*The name of the IMAP server.  If nil, uses the virtual server's name.")

(defvoo nnimap-server-address nil
  "Obsolete. Use `nnimap-address'.")

(defvoo nnimap-server-port nil
  "*The port of the IMAP server.  If nil, uses the default port. (143).")

(defvoo nnimap-imap-defs nil
  "*Definitions of variables to set up in the IMAP buffer.")

(defvoo nnimap-expunge-on-close 'always ; 'ask, 'never
  "When a IMAP group with articles marked for deletion is closed, this
variable determine if nnimap should actually remove the articles or
not.

If always, nnimap always perform a expunge when closing the group.
If never, nnimap never expunges articles marked for deletion.
If ask, nnimap will ask you if you wish to expunge marked articles.

When setting this variable to `never', you can only expunge articles
by using `G x' (gnus-group-nnimap-expunge) from the Group buffer.")

(defvoo nnimap-group-list-speed 'slow ; 'fast, 'medium
  "*If fast, do not show number of articles in the group list.
If medium, guess number of articles by using the UIDNEXT attribute.
If slow, fetch the UID of lowest/highest article.")

;; Splitting variables

(defvar nnimap-split-crosspost t
  "If non-nil, do crossposting if several split methods match the mail.
If nil, the first match found will be used.")

(defvar nnimap-split-inbox nil
  "*Name of mailbox to split mail from. Mail is read from this mailbox and
split according to rules in nnimap-split-rules.

This can be a string or a list of strings.")

(defvar nnimap-split-rule nil
  "*Mail will be split according to theese rules. Mail is read from mailbox(es)
specified in nnimap-split-inbox.

If you'd like, for instance, one mail group for mail from the
\"gnus-imap\" mailing list, one group for junk mail and leave
everything else in the incoming mailbox, you could do something like
this:

(setq nnimap-split-rule '((\"INBOX.gnus-imap\"   \"From:.*gnus-imap\")
			  (\"INBOX.junk\"        \"Subject:.*buy\")))

As you can see, nnimap-split-rule is a list of lists, where the first
element in each \"rule\" is the name of the IMAP mailbox, and the
second is a regexp that nnimap will try to match on the header to find
a fit.

The first element can also be a list. In that case, the first element
is the server the second element is the group on that server in which
the matching article will be stored.

The second element can also be a function.  In that case, it will be
called narrowed to the headers with the first element of the rule as
the argument.  It should return a non-nil value if it thinks that the
mail belongs in that group.")

;; Authorization / Privacy variables

(defvoo nnimap-auth-method nil
  "Obsolete.")

(defvoo nnimap-stream nil
  "How nnimap will connect to the server.

The default, nil, will try to use the \"best\" method the server can
handle.

Change this if

1) you want to connect with SSL. The SSL integration with IMAP is
   brain-dead so you'll have to tell it specifically.

2) your server is more capable than your environment -- i.e. your
   server accept Kerberos login's but you haven't installed the
   `imtest' program or your machine isn't configured for Kerberos.

Possible choices: kerberos4, ssl, network")

(defvoo nnimap-authenticator nil
  "How nnimap authenticate itself to the server.

The default, nil, will try to use the \"best\" method the server can
handle.

There is only one reason for fiddling with this variable, and that is
if your server is more capable than your environment -- i.e. you
connect to a server that accept Kerberos login's but you haven't
installed the `imtest' program or your machine isn't configured for
Kerberos.

Possible choices: kerberos4, cram-md5, login, anonymous.")

(defcustom nnimap-authinfo-file "~/.authinfo"
  "Authorization information for IMAP servers. In .netrc format."
  :type
  '(choice file
	   (repeat :tag "Entries"
		   :menu-tag "Inline"
		   (list :format "%v"
			 :value ("" ("login" . "") ("password" . ""))
			 (string :tag "Host")
			 (checklist :inline t
				    (cons :format "%v"
					  (const :format "" "login")
					  (string :format "Login: %v"))
				    (cons :format "%v"
					  (const :format "" "password")
					  (string :format "Password: %v")))))))


;; Internal variables.
(defvoo nnimap-need-expunge nil)
(defvoo nnimap-server-buffer nil)
(defvar nnimap-saved-info nil)
(defvoo nnimap-active-hashtb nil)

;; radford:
;;  shouldn't have to keep this info around, nnoo should???
;;  The only reason for this is `nnimap-server-opened' needs
;;  `nnimap-server-buffer' to do its work, but have to change
;;  servers to get the right one.  I don't think `opened'
;;  should be changing servers.  A better way anyone???
;; jas:
;;  i think this is the way to do it. from "Backend interface" in
;;  the gnus manual:
;;  "When Gnus asks for information from a backend--say nntp---on
;;  something, it will normally include a virtual server name in the
;;  function parameters. (If not, the backend should use the "current"
;;  virtual server.) For instance, nntp-request-list takes a virtual
;;  server as its only (optional) parameter. If this virtual server hasn't
;;  been opened, the function should fail."
(defvar nnimap-server-buffer-alist nil)

(defvar nnimap-debug "*nnimap-debug*")

(when nnimap-debug
  (require 'trace)
  (buffer-disable-undo (get-buffer-create nnimap-debug))
  (mapc (lambda (f) (trace-function-background f nnimap-debug))
        '(
          nnimap-close-group
          nnimap-close-server
          nnimap-open-server  
	  nnimap-expunge-close-group
	  nnimap-date-days-ago
	  nnimap-time-substract
          nnimap-possibly-change-group
          nnimap-possibly-change-server
	  nnimap-request-accept-article
          nnimap-request-article
          nnimap-request-article-part
          nnimap-request-body
          nnimap-request-close
          nnimap-request-create-group
          nnimap-request-delete-group
          nnimap-request-group
          nnimap-request-head
          nnimap-request-list
          nnimap-request-list-mapper
          nnimap-request-newgroups
          nnimap-request-post
          nnimap-request-rename-group
          nnimap-request-scan
;	  nnimap-send-command-wait
;	  nnimap-ok-p
	  nnimap-split-copy-delete-article
          nnimap-split-move-article
;	  nnimap-split-to-groups
	  nnimap-split-articles
          nnimap-request-update-info-internal
          nnimap-request-set-mark
          nnimap-retrieve-groups
          nnimap-retrieve-headers
          nnimap-server-opened
          nnimap-status-message
	  nnimap-update-alist-soft
	  nnimap-range-to-string
;          nnimap-request-expire-articles-progress
          nnimap-request-expire-articles
          nnimap-request-move-article
	  nnimap-mark-to-predicate
	  nnimap-mark-to-flag
	  nnimap-mark-permanent-p
	  gnus-group-nnimap-edit-acl-done
	  gnus-group-nnimap-edit-acl
	  nnimap-save-info-hook
	  nnimap-update-flags-hook
          )))


;;; Interface functions, required backend functions

;;; This seems to define the following:
;;;  `nnimap-close-server'       `nnimap-server-opened'
;;;  `nnimap-open-server'        `nnimap-status-message'
;;; open-server -> nnoo-change-server
;;; close-server -> nnoo-close-server
;;; server-opened -> nnoo-server-opened
;;; status-message -> nnoo-status-message

(nnoo-define-basics nnimap)

(defun nnimap-body-lines (body)
  "Return number of lines in article by looking at the mime bodystructure
BODY."
  (if (listp body)
      (if (stringp (car body))
	  (cond ((and (string= (car body) "TEXT")
		      (numberp (nth 7 body)))
		 (nth 7 body))
		((and (string= (car body) "MESSAGE")
		      (numberp (nth 9 body)))
		 (nth 9 body))
		(t 0))
	(apply '+ (mapcar 'nnimap-body-lines body)))
    0))

;; todo:
;; use NOV lines instead? A fetch like
;;   (UID RFC822.SIZE BODY BODY[HEADER.FIELDS (References)]) would do it
;; remove redundant lines:/chars:
(deffoo nnimap-retrieve-headers (articles &optional group server fetch-old)
  (when (nnimap-possibly-change-group group server)
    (with-current-buffer nntp-server-buffer (erase-buffer))
    (let* ((uncompressed (gnus-uncompress-sequence articles))
          (compressed (gnus-compress-sequence uncompressed t)))
      (with-current-buffer nnimap-server-buffer
	;; Reset message info, this makes sure we return 423 on
	;; articles that were removed by another client while we
	;; had the mailbox SELECTed. This isn't really necessery (the user
	;; will find out when he selects the article anyway).
	;(imap-message-reset)
	(nnimap-ok-p (nnimap-send-command-wait
		      (concat "UID FETCH "
			      (if (and fetch-old (not (numberp fetch-old)))
				  "1:*"
				(nnimap-range-to-string compressed))
			      " (UID RFC822.HEADER RFC822.SIZE BODY)")))
	(mapc (lambda (num)
		(let* ((header (imap-message-get num 'RFC822.HEADER))
		       (size   (imap-message-get num 'RFC822.SIZE))
		       (body   (imap-message-get num 'BODY))
		       (lines  (nnimap-body-lines body)))
		  (with-current-buffer nntp-server-buffer
		    (if (not header)
			(insert "423 Bad article number.\n")
		      (insert (format "221 %d Article retrieved.\n" num))
		      (insert (format "Chars: %d\n" size))
		      (insert (format "Lines: %d\n" lines))
		      (insert header)
		      (delete-char -1)
		      (insert ".\n")))))
	      uncompressed)
	'headers))))

(deffoo nnimap-open-server (server &optional defs)
  (or (and (nnimap-server-opened server)
	   (nnoo-change-server 'nnimap server defs)
           (if (null nnimap-server-buffer)
               (error "this shouldn't happen")
             t))
      (progn
        (unless (assq 'nnimap-address defs)
	  (if (assq 'nnimap-server-address defs)
	      (push (list 'nnimap-address (cadr (assq 'nnimap-server-address
						      defs))) defs)
	    (push (list 'nnimap-address server) defs)))
        (unless (assq 'nnimap-server-buffer defs)
          (push (list 'nnimap-server-buffer (concat " *nnimap* " server))
		defs))
	(nnoo-change-server 'nnimap server defs)
	(if (not (imap-open nnimap-address nnimap-server-port
			    nnimap-stream nnimap-authenticator 
			    nnimap-server-buffer))
	    (nnheader-report 'nnimap "Could not connect to server %s" server)
	  (unless (or (imap-capability 'IMAP4 nnimap-server-buffer)
		      (imap-capability 'IMAP4rev1 nnimap-server-buffer)
		      (imap-capability 'IMAP4REV1 nnimap-server-buffer))
	    (imap-close nnimap-server-buffer)
	    (nnheader-report 'nnimap "Server %s not IMAP4" server))
	  (let (list alist user passwd)
	    (and (fboundp 'gnus-parse-netrc)
		 (setq list (gnus-parse-netrc nnimap-authinfo-file)
		       alist (or (and (gnus-netrc-get (gnus-netrc-machine 
						       list server) "machine")
				      (gnus-netrc-machine list server))
				 (gnus-netrc-machine list nnimap-address))
		       user (gnus-netrc-get alist "login")
		       passwd (gnus-netrc-get alist "password")))
	    (if (imap-authenticate nnimap-server-buffer user passwd)
		(push (cons server nnimap-server-buffer)
		      nnimap-server-buffer-alist)
	      (imap-close nnimap-server-buffer)
	      (nnheader-report 'nnimap "Could not authenticate to %s"
			       server)))))))

(deffoo nnimap-close-server (&optional server)
  (let ((s-b (assoc (setq server (or server (nnoo-current-server 'nnimap)))
		    nnimap-server-buffer-alist)))
    (when s-b
      (setq nnimap-server-buffer nil)
      (setq nnimap-server-buffer-alist (delq s-b nnimap-server-buffer-alist))
      (imap-close (cdr s-b)))
    (nnoo-close-server 'nnimap server)))

(deffoo nnimap-request-close ()
  (mapc (lambda (s-b) (nnimap-close-server (car s-b)))
        nnimap-server-buffer-alist)
  (setq nnimap-server-buffer-alist nil))

(deffoo nnimap-server-opened (&optional server)
  (let ((buffer (cdr (assoc 
                      (setq server (or server (nnoo-current-server 'nnimap)))
                      nnimap-server-buffer-alist))))
    (if (and (gnus-buffer-live-p buffer)
	     (gnus-buffer-live-p nntp-server-buffer))
	(let ((running (imap-opened buffer)))
	  ;; clean up autologouts
	  (unless running
	    (nnimap-close-server server))
	  running))))

(deffoo nnimap-status-message (&optional server)
  (let ((buffer (cdr (assoc 
                      (setq server (or server (nnoo-current-server 'nnimap)))
                      nnimap-server-buffer-alist))))
    (when buffer
      (with-current-buffer buffer
	(or (cdr imap-last-status)
	    (nnoo-status-message 'nnimap server))))))

(deffoo nnimap-request-article (article &optional group server to-buffer)
  (nnimap-request-article-part 'RFC822 article group server to-buffer t))
(deffoo nnimap-request-head    (article &optional group server to-buffer)
  (nnimap-request-article-part 'RFC822.HEADER article group server to-buffer))
(deffoo nnimap-request-body    (article &optional group server to-buffer)
  (nnimap-request-article-part 'RFC822.TEXT article group server to-buffer t))

(defun nnimap-request-article-part (part article &optional 
					 group server to-buffer add-peek)
  (when (nnimap-possibly-change-group group server)
    (with-current-buffer nnimap-server-buffer
      (when (stringp article)
        ;; Find the article number by Message-Id
        (let ((articles (imap-search (format "HEADER Message-Id %d" article))))
          (when (cdr articles)
            (message 
             "IMAP Warning: Message Id search yielded more than one article %s"
             articles))
          (setq article (car articles))))
      (when article
        ;; Find the article by number
        (nnimap-send-command-wait (format "UID FETCH %d (%s%s)" article part
					  (if add-peek ".PEEK" "")))
        (let ((text (funcall (if (and (fboundp 'string-as-multibyte)
				      (subrp (symbol-function 
					      'string-as-multibyte)))
				 'string-as-multibyte
			       'identity) (imap-message-get article part))))
          (with-current-buffer (or to-buffer nntp-server-buffer)
            (erase-buffer)
            (if (not text)
                (nnheader-report 'nnimap 
				 (format "Article %s does not exist." article))
              (insert text)
	      t)))))))

;;; Select GROUP and unless FAST return 211 EXISTS LOWEST HIGHEST GROUP
;;; The active info returned goes into the `gnus-active-hashtb'.
;;; Don't call this directly, call `gnus-activate-group'.
;;; `gnus-activate-group' can call `request-scan' before request-group.
;;; Ok, here seems to be the problem.  We return 'group and
;;; `gnus-groups-to-gnus-format' seem to not use a prefix "nnimap+yoyo:"
;;; when entering something into the active-hashtb but it does when
;;; it looks for new news.  Damn.
(deffoo nnimap-request-group (group &optional server fast)
  (nnimap-request-update-info-internal
   group
   (gnus-get-info (gnus-group-normally-qualified 'nnimap server group))
   server)
  (if fast
      (with-current-buffer nntp-server-buffer 
	(erase-buffer)
	t)
    (gnus-message 7 "Opening nnimap group %s..." group)
    (when (nnimap-possibly-change-group group server)
      (with-current-buffer nnimap-server-buffer
	;; clear message data, we won't necesserily have to do this if
	;; it weren't for buggy CCmail (we can't know how many tagged
	;; responses were returned otherwise).
	(imap-message-reset)
	(let ((exists (imap-folder-get 'EXISTS))
	      articles)
	  (if (eq 0 exists)
	      (setq articles '(0))
	    (when (nnimap-ok-p (nnimap-send-command-wait "FETCH 1,* (UID)"))
	      (imap-message-map (lambda (uid Uid)
				  (push uid articles)) 'UID)
	      ;; start of bug workaround code
	      ;; Lotus CCmail is broken, returns:
	      ;; C13 FETCH 1,* (UID)
	      ;; * 1 FETCH (UID 4198)
	      ;; C13 OK Completed
	      ;; we fetch * in addition
	      (when (< 2 (length articles))
		(when (nnimap-ok-p (nnimap-send-command-wait "FETCH * (UID"))
		  (imap-message-map (lambda (uid Uid)
				      (push uid articles)) 'UID)))
	      ;; end of bug workaround code
	      ))
	  (when articles
	    (nnheader-insert "211 %d %d %d %s\n" exists
			    (max 1 (apply 'min articles))
			    (apply 'max articles) group)))))
    (gnus-message 7 "Opening nnimap group %s...done" group)))

;; Note that there is no need for this in current Gnus (5.6.27), all
;; you need to do is use gnus-group-prefixed-name.  I'm not sure when
;; this got fixed.  That's what I get for not using the current
;; version.
(defun gnus-group-normally-qualified (backend server group)
  ;; This is the format for groups in the group-info.
  ;; native groups are not qualified.  Why?
  (let ((method (gnus-server-to-method (format "%s:%s" backend server))))
    (if (gnus-method-equal gnus-select-method method)
        ;; Native (we of course can't call `gnus-native-p' since it
        ;; assumes group is the result of this function
        group
      (gnus-group-prefixed-name group method))))
            
(deffoo nnimap-close-group (group &optional server)
  (when (nnimap-possibly-change-group group server)
    (nnimap-expunge-close-group server)))

;;; Returns: GROUP HIGHEST LOWEST [ynmxj=]
(defun nnimap-request-list-mapper (group)
  (unless (or (member '\\NoSelect (imap-folder-get 'flags group))
	      ;; We ignore groups with spaces (Gnus can't handle them)
	      (string-match " " group))
    (let (high)
      (gnus-message 7 "Generating active list for nnimap group %s" group)
      (cond 
       ((eq nnimap-group-list-speed 'slow)
	(when (imap-mailbox-select group)
	  (let ((exists (imap-folder-get 'EXISTS))
		articles)
	    (if (eq 0 exists)
		(with-current-buffer nntp-server-buffer
		  (insert (format "%s 0 1 y\n" group))
		  t)
	      ;; if it weren't for buggy CCmail we needn't reset
	      (imap-message-reset)
	      (when (nnimap-ok-p (nnimap-send-command-wait "FETCH 1,* (UID)"))
		(imap-message-map (lambda (uid Uid)
				    (push uid articles)) 'UID)
		;; start of bug workaround code
		;; Lotus CCmail is broken, returns:
		;; C13 FETCH 1,* (UID)
		;; * 1 FETCH (UID 4198)
		;; C13 OK Completed
		;; we fetch * in addition
		(when (< 2 (length articles))
		  (when (nnimap-ok-p (nnimap-send-command-wait "FETCH * (UID"))
		    (imap-message-map (lambda (uid Uid)
					(push uid articles)) 'UID)))
		;; end of bug workaround code
		(when articles
		  (with-current-buffer nntp-server-buffer
		    (insert (format "%s %d %d y\n" group
				    (apply 'max articles)
				    (apply 'min articles)))
		    t)))))))
       ((eq nnimap-group-list-speed 'medium)
	(when (nnimap-ok-p (nnimap-send-command-wait
			    (concat "STATUS " group 
				    " (UIDNEXT)")))
	  (setq high (1- (imap-folder-get 'UIDNEXT group)))
	  (with-current-buffer nntp-server-buffer
	    (insert (format "%s %d 1 y\n" group high))
	    t)))
       ((eq nnimap-group-list-speed 'fast)
	(with-current-buffer nntp-server-buffer
	  (insert (format "%s 0 1 y\n" group))
	  t))
       (t
	(error "Unknown nnimap-group-list-speed: %s"
	       nnimap-group-list-speed))))))

(defun nnimap-pattern-to-list-arguments (pattern)
  (mapcar (lambda (p) (if (consp p)
			  (cons (concat "\"" (car p) "\"")
				(concat "\"" (cdr p) "\""))
			(cons "\"\"" (concat "\"" p "\""))))
	  (if (and (listp pattern)
		   (listp (cdr pattern)))
	      pattern
	    (list pattern))))

;;; Returns: GROUP HIGHEST LOWEST [ynmxj=]
(deffoo nnimap-request-list (&optional server)
  (when (nnimap-possibly-change-server server)
    (with-current-buffer nntp-server-buffer (erase-buffer))
    (with-current-buffer nnimap-server-buffer
      (let* ((patterns (nnimap-pattern-to-list-arguments nnimap-list-pattern))
	     pattern)
	(gnus-message 5 "Generating active list for %s" server)
	(while (setq pattern (pop patterns))
	  (nnimap-ok-p (nnimap-send-command-wait
                        (concat nnimap-list-method " "
                                (car pattern) " "
                                (cdr pattern))))))
      (let ((nnimap-group-list-speed 'fast))
	(imap-mailbox-map 'nnimap-request-list-mapper))
      t)))
  

;;; IMAP doesn't support posting, but this must be defined
(deffoo nnimap-request-post (&optional server)
  nil)


;;; Interface functions, optional backend functions

;; This gets called instead of `nnimap-request-list' when
;; `gnus-read-active-file' is 'some instead of t.
;; Returns: GROUP HIGHEST LOWEST [ynmxj=] ???
(deffoo nnimap-retrieve-groups (groups &optional server)
  (when (nnimap-possibly-change-server server)
    (with-current-buffer nntp-server-buffer (erase-buffer))
    (with-current-buffer nnimap-server-buffer
      ;; Force the slow method for now since this will only be called
      ;; for subscribed groups. 
      (let (group)
        (gnus-message 5 "Generating active info for certain groups on %s"
                      server)
        (while (setq group (pop groups))
          (nnimap-request-list-mapper group)))
      'active)))

;; This is nnimap-request-update-info, but it's so extremely slow
;; we can't have Gnus call it all the time. Instead, it's called by
;; nnimap-request-group.
(deffoo nnimap-request-update-info-internal (group info &optional server)
  ;; We reset the uidvalidity here because we are about to do a full resync.
  (when info ;; nil info when group doesn't exist or has just been created.
             ;; Gnus will create it for us later.
    (gnus-info-set-params info (nnimap-update-alist-soft
                               'uidvalidity nil (gnus-info-params info)) t)
  (when (nnimap-possibly-change-group group server) ;; SELECT
    (with-current-buffer nnimap-server-buffer
      (gnus-message 5 "Updating info for mailbox %s" group)
      
      ;; Replace list of read and marked articles with authoritative
      ;; data from server.
      
      ;; If server is read-only (no article mark memory), we don't
      ;; overwrite local flags -- this way one can use it as a nntp type
      ;; of server (all mark memory in client).

      (when (nnimap-mark-permanent-p 'read)
	(gnus-info-set-read 
         info
         (let* (;; oldseen could contain articles marked unread by other
                ;; imap clients!  we correct this
                (oldseentmp (gnus-uncompress-range (gnus-info-read info)))
                (unseen (imap-search "UNSEEN UNDELETED"))
                (oldseen (gnus-set-difference oldseentmp unseen))
                ;; oldseen might lack articles marked as read by other
                ;; imap clients! we correct this
                (newseen (imap-search "SEEN"))
                ;; ok, read articles are in oldseen+newseen (xxx lots of dupes)
                (seen (append oldseen newseen))
                ;; sort to have gnus-compress-sequence remove dupes
                (seens (sort seen '<))
                (read (gnus-compress-sequence seens t)))
	   ;; we can't return '(1) since this isn't a "list of ranges",
	   ;; and we can't return '((1)) since gnus-list-of-unread-articles
	   ;; is buggy so we return '((1 . 1)).
           (if (and (integerp (car read))
		    (null (cdr read)))
               (list (cons (car read) (car read)))
             read))))

      (mapc (lambda (pred)
	      (when (and (nnimap-mark-permanent-p (cdr pred))
			 (member (nnimap-mark-to-flag (cdr pred))
				 (imap-folder-get 'list-flags)))
		(gnus-info-set-marks
		 info
		 (nnimap-update-alist-soft
		  (cdr pred)
		  (gnus-compress-sequence
		   (imap-search (nnimap-mark-to-predicate (cdr pred))))
		  (gnus-info-marks info))
		 t)))
	      gnus-article-mark-lists))))
  info)

;; Respond to articles with mail
(deffoo nnimap-request-type (group article)
  'mail)

(defun nnimap-split-copy-delete-article (article group to-group server)
  "Move article ARTICLE from group GROUP on current server to group TO-GROUP."
  (when (nnimap-ok-p (nnimap-send-command-wait
		      (format "UID COPY %d %s" article to-group)))
    (setq nnimap-need-expunge t)
    (if (imap-message-flags-add (format "%d" article) "\\Seen \\Deleted")
	(message "IMAP split moved %s:%s:%d to %s" server group
		 article to-group)
      (error "IMAP flag store failed: you may have unread mail marked as read!"))))

(defun nnimap-split-move-article (article group to-group server)
  (when to-group
    (unless (nnimap-split-copy-delete-article article group to-group server)
      (message "Could not find mailbox %s, creating..." to-group)
      (if (nnimap-ok-p (nnimap-send-command-wait
			(format "CREATE %s" to-group)))
	  (nnimap-split-copy-delete-article article group to-group server)
	(message "Could not create mailbox %s: %s" 
		 to-group imap-last-status)))))

;; tries to match all rules in nnimap-split-rule against content of 
;; nntp-server-buffer, returns a list of groups that matched.
(defun nnimap-split-to-groups (rule)
  (let (to-groups)
    (with-current-buffer nntp-server-buffer
      (mapcar (lambda (rule) 
		(let ((group (car rule))
		      (regexp (cadr rule)))
		  ;; Fold continuation lines.
		  (goto-char (point-min))
		  (while (re-search-forward "\\(\r?\n[ \t]+\\)+" nil t)
		    (replace-match " " t t))
		  (goto-char (point-min))
		  (when (re-search-forward regexp nil t)
		    (setq to-groups (cons group to-groups)))))
	      rule)
      (reverse to-groups))))

(defun nnimap-split-find-rule (server inbox)
  nnimap-split-rule)

(defun nnimap-split-find-inbox (server)
  (if (listp nnimap-split-inbox)
      nnimap-split-inbox
    (list nnimap-split-inbox)))

(defun nnimap-split-articles (&optional group server)
  (when (nnimap-possibly-change-server server)
    (with-current-buffer nnimap-server-buffer
      (let (rule inbox (inboxes (nnimap-split-find-inbox server)))
	;; iterate over inboxes
	(while (and (setq inbox (pop inboxes))
		    (nnimap-possibly-change-group inbox)) ;; SELECT
	  ;; find split rule for this server / inbox
	  (when (setq rule (nnimap-split-find-rule server inbox))
	    (let (article (unseens (imap-search "UNSEEN")))
	      ;; iterate over articles
	      (while (setq article (pop unseens))
		(when (nnimap-request-head article)
		  ;; article into what groups?
		  (let ((groups (nnimap-split-to-groups rule)))
		    ;; move it there
		    (if nnimap-split-crosspost
			;; move to all boxes
			(let (to-group)
			  (while (setq to-group (pop groups))
			    (nnimap-split-move-article article inbox 
						       to-group server)))
		      ;; move to first matching box, if any
		      (nnimap-split-move-article article inbox 
						 (car groups) server)))))))
	  (when (imap-mailbox-select inbox) ;; just in case
	    ;; todo: UID EXPUNGE (if available) to remove splitted articles
	    (nnimap-expunge-close-group)))
	t))))

;; nnimap-request-scan doesn't need to do anything. the delivery agent
;; program is responsible for putting new message in the imap folders.
;; compare the situation with nntp (nil function) and with all nnmail-
;; backends (fetches mail from spools, POPing, saving in files on local
;; storage etc).
;; On the other hand, we could do message splitting here.
(deffoo nnimap-request-scan (&optional group server)
  (nnimap-split-articles group server))

;; until everyone uses gnus >= 5.6.24
(deffoo nnimap-request-group-description (group &optional server)
  (when (nnimap-possibly-change-server server)
    (with-current-buffer nntp-server-buffer
      (erase-buffer))
    t))
 
;; until everyone uses gnus >= 5.6.24
(deffoo nnimap-request-list-newsgroups (&optional server)
  (when (nnimap-possibly-change-server server)
    (with-current-buffer nntp-server-buffer
      (erase-buffer))
    t))

;; until everyone uses gnus >= 5.6.24
(deffoo nnimap-request-newgroups (date &optional server)
  (when (nnimap-possibly-change-server server)
    (with-current-buffer nntp-server-buffer
      (erase-buffer))
    t))

;; (nn)IMAP specific decisions:
;;
;; o deletion of reply-marks is prohibited
;; o dormant articles are also marked as ticked
;;
;; action looks like:
;;   (((1 . 10) 'set '(read ticked))
;;    ((1 . 10) 'del '(tick reply expire killed dormant save download unsend)))
;;
(deffoo nnimap-request-set-mark (group actions &optional server)
  (when (nnimap-possibly-change-group group server)
    (with-current-buffer nnimap-server-buffer
      (let (action)
	(gnus-message 7 "Setting marks in %s:%s..."  
		      (nnoo-current-server 'nnimap) group)
	(while (setq action (pop actions))
	  (let ((range (nth 0 action))
		(what  (nth 1 action))
		(pred  (nth 2 action)))
	    ;; enforce local decisions
	    (if (eq what 'del)
		(setq pred (delq 'reply pred)))
	    (if (memq 'dormant pred)
		(setq pred (cons 'tick pred)))
	    (when (and range pred)
	      (cond ((eq what 'del)
		     (imap-message-flags-del (nnimap-range-to-string range)
					   (nnimap-mark-to-flag pred nil t)))
		    ((eq what 'add)
		     (imap-message-flags-add (nnimap-range-to-string range)
					   (nnimap-mark-to-flag pred nil t)))
		    ((eq what 'set)
		     (imap-message-flags-set (nnimap-range-to-string range)
					   (nnimap-mark-to-flag pred nil t)))))))
	(gnus-message 7 "Setting marks in %s:%s...done" 
		      (nnoo-current-server 'nnimap) group)))))

(deffoo nnimap-request-create-group (group &optional server args)
  (when (nnimap-possibly-change-server server)
    (nnimap-ok-p (nnimap-send-command-wait
                  (concat "CREATE " group) 
                  nnimap-server-buffer))))

(defun nnimap-time-substract (time1 time2)
  "Return TIME for TIME1 - TIME2."
  (let* ((ms (- (car time1) (car time2)))
	 (ls (- (nth 1 time1) (nth 1 time2))))
    (if (< ls 0)
	(list (- ms 1) (+ (expt 2 16) ls))
      (list ms ls))))

(defun nnimap-date-days-ago (daysago)
  "Return date, in format \"3-Aug-98\", for DAYSAGO days ago."
  (let ((date (format-time-string "%d-%b-%y" 
				  (nnimap-time-substract
				   (current-time)
				   (if (fboundp 'days-to-time)
				       (days-to-time daysago)
				     (nnmail-days-to-time daysago))))))
    (if (eq ?0 (string-to-char date))
	(substring date 1)
      date)))

(defun nnimap-request-expire-articles-progress (num fetch data)
  (gnus-message 5 "Expiring; marking article %d for deletion..." num))

;; Notice that we don't actually delete anything, we just mark them deleted.
(deffoo nnimap-request-expire-articles (articles group &optional server force)
  (let (oldarts (artseq (gnus-compress-sequence articles)))
    (when (and artseq (nnimap-possibly-change-group group server))
      (with-current-buffer nnimap-server-buffer
	(if force
	    ;; add delete flag to article
	    (when (imap-message-flags-add (nnimap-range-to-string artseq) 
					"\\Deleted")
	      (setq nnimap-need-expunge t)
	      (setq articles nil))
	  (let ((days (or (and nnmail-expiry-wait-function
			       (funcall nnmail-expiry-wait-function group))
			  nnmail-expiry-wait)))
	    (cond ((eq days 'immediate)
		   ;; add delete flag to article
		   (when (imap-message-flags-add 
			  (nnimap-range-to-string artseq) "\\Deleted")
		     (setq nnimap-need-expunge t)
		     (setq articles nil)))
		  ((numberp days)
		   ;; We should not search only gnus-expired articles,
		   ;; Gnus makes sure request-expire-articles is called
		   ;; with correct arguments. (with total-expire,
		   ;; the articles won't have gnus-expire set but should
		   ;; be expired)
		   (setq oldarts (imap-search
				  (format "UID %s NOT SINCE %s"
					  (nnimap-range-to-string artseq)
					  (nnimap-date-days-ago days))))
		   (let ((imap-cb-fetch-hook 
			  'nnimap-request-expire-articles-progress))
		     (when (and oldarts (imap-message-flags-add 
					 (nnimap-range-to-string 
					  (gnus-compress-sequence oldarts))
					 "\\Deleted"))
		       (setq nnimap-need-expunge t)
		       (setq articles (gnus-set-difference articles
							   oldarts)))))))))))
  ;; return articles not deleted
  articles)

(deffoo nnimap-request-move-article (article group server
					     accept-form &optional last)
  (save-excursion
    (let ((buf (get-buffer-create " *nnimap move*"))
	  result)
      (and 
       (nnimap-request-article article group server)
       (save-excursion
	 (set-buffer buf)
	 (buffer-disable-undo (current-buffer))
	 (insert-buffer-substring nntp-server-buffer)
	 (setq result (eval accept-form))
	 (kill-buffer buf)
	 result)
       (nnimap-request-expire-articles (list article) group server t))
      result)))

(deffoo nnimap-request-accept-article (group &optional server last)
  (when (nnimap-possibly-change-server server)
    ;; We assume article is appended as UIDNEXT if no UIDPLUS support.
    (when (or (imap-capability 'UIDPLUS nnimap-server-buffer)
	      (nnimap-ok-p (nnimap-send-command-wait
			    (concat "STATUS " group " (UIDNEXT)")
			    nnimap-server-buffer)))
      (with-current-buffer (current-buffer)
	(goto-char (point-min))
	(unless (string= "\n" imap-eol)
	  (while (re-search-forward "\n" nil t)
	    (replace-match imap-eol))))
      (when (nnimap-ok-p (nnimap-send-command-wait 
			  ;; Optional flags,date???
			  (list (concat "APPEND " group " ") 
				(current-buffer))
			  nnimap-server-buffer))
	(let ((high (if (imap-capability 'UIDPLUS nnimap-server-buffer)
			(cdr (imap-folder-get 'appenduid nil 
					      nnimap-server-buffer))
		      (imap-folder-get 'UIDNEXT group
				       nnimap-server-buffer))))
	  (when high
	    (cons group high)))))))

;; (deffoo nnimap-request-replace-article -- IMAP does not support replacing

(deffoo nnimap-request-delete-group (group force &optional server)
  (when (nnimap-possibly-change-server server)
    (nnimap-ok-p (nnimap-send-command-wait
                  (concat "DELETE " group)
                  nnimap-server-buffer))))

(deffoo nnimap-request-rename-group (group new-name &optional server)
  (when (nnimap-possibly-change-server server)
    (when (string= "INBOX" group)
      (nnheader-report 
       'nnimap 
       "Renaming INBOX will only move content to a newly created mailbox")
      (sit-for 1))
    (nnimap-ok-p (nnimap-send-command-wait
                  (concat "RENAME " group " " new-name)
                  nnimap-server-buffer))))


;;; Internal functions

;;
;; This is confusing.
;;
;; mark      => read, tick, draft, reply etc
;; flag      => "\\Seen", "\\Flagged", "\\Draft", "gnus-expire" etc
;; predicate => "SEEN", "FLAGGED", "DRAFT", "KEYWORD gnus-expire" etc
;;
;; Mark should not really contain 'read since it's not a "mark" in the Gnus
;; world, but we cheat. Mark == gnus-articlemark-lists + '(read . read).
;;

(defconst nnimap-mark-to-predicate-alist
  (mapcar 
   (lambda (pair) ; cdr is the mark
     (or (assoc (cdr pair)
                '((read . "SEEN")
                  (tick . "FLAGGED")
                  (draft . "DRAFT")
                  (reply . "ANSWERED")))
         (cons (cdr pair)
               (format "KEYWORD gnus-%s" (symbol-name (cdr pair))))))
   (cons '(read . read) gnus-article-mark-lists)))

(defun nnimap-mark-to-predicate (pred)
  "Convert a Gnus mark (a symbol such as read, tick, expire) to a IMAP
predicate (a string such as \"SEEN\", \"FLAGGED\", \"KEYWORD
gnus-expire\") to be used within a IMAP SEARCH query."
  (cdr (assq pred nnimap-mark-to-predicate-alist)))

(defconst nnimap-mark-to-flag-alist 
  (mapcar 
   (lambda (pair)
     (or (assoc (cdr pair)
                '((read . "\\Seen")
                  (tick . "\\Flagged")
                  (draft . "\\Draft")
                  (reply . "\\Answered")))
         (cons (cdr pair)
               (format "gnus-%s" (symbol-name (cdr pair))))))
   (cons '(read . read) gnus-article-mark-lists)))

(defun nnimap-mark-to-flag-1 (preds)
  (if (and (not (null preds)) (listp preds))
      (cons (nnimap-mark-to-flag (car preds))
	    (nnimap-mark-to-flag (cdr preds)))
    (cdr (assoc preds nnimap-mark-to-flag-alist))))

(defun nnimap-mark-to-flag (preds &optional always-list make-string)
  "Convert a Gnus mark (a symbol such as read, tick, expire) to a IMAP
flag (a string such as \"\\Seen\", \"\\Flagged\", \"gnus-expire\") to
be used in a STORE FLAGS command."
  (let ((result (nnimap-mark-to-flag-1 preds)))
    (setq result (if (and (or make-string always-list)
			  (not (listp result)))
		     (list result)
		   result))
    (if make-string
	(mapconcat (lambda (flag)
		     (if (listp flag)
			 (mapconcat 'identity flag " ")
		       flag))
		   result " ")
      result)))

(defun nnimap-mark-permanent-p (mark &optional group)
  "Return t iff MARK can be permanently (between IMAP sessions) saved
on articles, in GROUP."
  (with-current-buffer nnimap-server-buffer
    (or (member "\\*" (imap-folder-get 'permanentflags group))
	(member (nnimap-mark-to-flag mark) (imap-folder-get 'permanentflags
							    group)))))
  
(defun nnimap-update-alist-soft (key value alist)
  (if value 
      (cons (cons key value) (remassoc key alist))
    (remassoc key alist)))

(defun nnimap-range-to-string (range)
  (mapconcat 
   (lambda (item)
     (if (consp item)
         (format "%d:%d" 
                 (car item) (cdr item))
       (format "%d" item)))
   (if (and (listp range) (not (listp (cdr range))))
       (list range) ;; make (1 . 2) into ((1 . 2))
     range)
   ","))

(defun nnimap-send-command-wait (command &optional buffer)
  (unless (and (stringp command) (string-match "%" command))
    (gnus-message 10 (apply 'concat
			    (if (listp command)
				(mapcar
				 (lambda (s-b) 
				   (cond 
				    ((bufferp s-b) (buffer-name s-b))
				    ((functionp s-b) "<func>")
				    (s-b)))
				 command)
			      (list command)))))
  (imap-send-command-wait command buffer))

(defun nnimap-ok-p (status)
  (if status
      (if (imap-ok-p status)
	  t
	(nnheader-report 'nnimap (cdr status)))
    (nnheader-report 'nnimap (format "IMAP Command Timed Out"))))

(defun nnimap-expunge-close-group (&optional server)
  (with-current-buffer nnimap-server-buffer
    (when (and (nnimap-possibly-change-server server)
	       imap-current-folder)
      (cond ((eq nnimap-expunge-on-close 'always)
	     (when nnimap-need-expunge 
	       (setq nnimap-need-expunge nil)
	       (imap-send-command "EXPUNGE"))
	     (imap-mailbox-close))
	    ((eq nnimap-expunge-on-close 'never)
	     (imap-mailbox-unselect))
	    ((eq nnimap-expunge-on-close 'ask)
	     (when (imap-search "DELETED")
	       (if (gnus-y-or-n-p (format "Expunge articles in group `%s'? "
					  imap-current-folder))
		   (and (nnimap-ok-p (nnimap-send-command-wait "EXPUNGE"))
			(imap-mailbox-close))
		 (imap-mailbox-unselect)))))))
  (not imap-current-folder))

(defun nnimap-possibly-change-server (server)
  "Change to server SERVER if needed (open it if it's closed). If SERVER is
nil, change to current server."
  (and (setq server (or server (nnoo-current-server 'nnimap)))
       (nnimap-open-server server)))

(defun nnimap-possibly-change-group (group &optional server)
  (setq server (or server (nnoo-current-server 'nnimap)))
  (when (nnimap-possibly-change-server server)
    (with-current-buffer nnimap-server-buffer
      (when group
	(let ((groupname (gnus-group-normally-qualified 
			  'nnimap server group)))
	  (if (and imap-current-folder
		   (not (string= group imap-current-folder)))
	      (nnimap-expunge-close-group))
	  (when (imap-mailbox-select group nil 
				    (gnus-ephemeral-group-p groupname))
	    ;; check/set UIDVALIDITY
	    (let ((new-uid (imap-folder-get 'uidvalidity))
		  (old-uid (gnus-group-get-parameter groupname 'uidvalidity))
		  (info (gnus-get-info groupname)))
	      (if (not old-uid)
		  (gnus-group-add-parameter groupname (cons 'uidvalidity 
							    new-uid))
		(when (not (equal old-uid new-uid))
		  (if (and (not (gnus-info-marks info))
			   (not (gnus-info-read info)))
		      (gnus-group-set-parameter groupname 'uidvalidity 
						new-uid)
		    (message "UIDVALIDITY clash. Old value `%s', new `%s'"
			     old-uid new-uid)
		    (imap-mailbox-unselect))))))))
      imap-current-folder)))

;;; Gnus functions

(defun gnus-group-nnimap-expunge (group)
  "Expunge deleted articles in current nnimap GROUP."
  (interactive (list (gnus-group-group-name)))
  (let ((mailbox (gnus-group-real-name group))
	method)
    (unless group
      (error "No group on current line"))
    (unless (gnus-get-info group)
      (error "Killed group; can't be edited"))
    (unless (eq 'nnimap (car (setq method (gnus-find-method-for-group group))))
      (error "Expunging only available for nnimap groups"))
    (when (nnimap-possibly-change-group mailbox (cadr method))
      (nnimap-send-command-wait "EXPUNGE" nnimap-server-buffer))))

(defun gnus-group-nnimap-edit-acl (group)
  "Edit the Access Control List of current nnimap GROUP."
  (interactive (list (gnus-group-group-name)))
  (let ((mailbox (gnus-group-real-name group))
	method acl)
    (unless group
      (error "No group on current line"))
    (unless (gnus-get-info group)
      (error "Killed group; can't be edited"))
    (unless (eq 'nnimap (car (setq method (gnus-find-method-for-group group))))
      (error "ACL editing only available for nnimap groups"))
    (when (nnimap-possibly-change-server (cadr method))
      (unless (imap-capability 'ACL nnimap-server-buffer)
	(error "Your server does not support ACL editing"))
      (gnus-edit-form (with-current-buffer nnimap-server-buffer
			(imap-folder-set 'acl nil mailbox)
			(nnimap-send-command-wait (format "GETACL %s" mailbox))
			(setq acl (destructive-plist-to-alist 
				   (imap-folder-get 'acl mailbox))))
		      (format "Editing the access control list for `%s'.

   An access control list is a list of (identifier . rights) elements.

   The identifier specifies the corresponding user. The identifier
   `anyone' is reserved to refer to the universal identity.

   Rights is a string listing a (possibly empty) set of alphanumeric
   characters, each character listing a set of operations which is being
   controlled. Letters are reserved for ``standard'' rights, listed
   below.  Digits are reserved for implementation or site defined rights.

   l - lookup (mailbox is visible to LIST/LSUB commands)
   r - read (SELECT the mailbox, perform CHECK, FETCH, PARTIAL,
       SEARCH, COPY from mailbox)
   s - keep seen/unseen information across sessions (STORE SEEN flag)
   w - write (STORE flags other than SEEN and DELETED)
   i - insert (perform APPEND, COPY into mailbox)
   p - post (send mail to submission address for mailbox,
       not enforced by IMAP4 itself)
   c - create (CREATE new sub-mailboxes in any implementation-defined
       hierarchy)
   d - delete (STORE DELETED flag, perform EXPUNGE)
   a - administer (perform SETACL)" group)
		      `(lambda (form) 
			 (gnus-group-nnimap-edit-acl-done 
			  ,mailbox ',method ',acl form))))))

(defun gnus-group-nnimap-edit-acl-done (mailbox method old-acls acls)
  (when (nnimap-possibly-change-server (cadr method))
    (with-current-buffer nnimap-server-buffer
      ;; delete all removed identifiers
      (let ((deleted (copy-list old-acls))
	    (dontdelete acls) acl)
	(while (setq acl (pop deleted))
	  (unless (assoc (car acl) dontdelete)
	    (let ((status 
		   (nnimap-send-command-wait
		    (format "DELETEACL %s %s" mailbox (car acl)))))
	      (when (and (listp status)
			 (eq 'NO (car status)))
		(error "Can't delete ACL: %s" (cadr status))))))
      ;; set all changed acl's
      (let ((new-acls acls) acl)
	(while (setq acl (pop new-acls))
	  (let* ((user (car acl))
		 (access (cdr acl))
		 (old-access (cdr (assoc user old-acls))))
	    (unless (string= access old-access)
	      (let ((status 
		     (nnimap-send-command-wait
		      (format "SETACL %s %s %s" mailbox user access))))
		(when (and (listp status)
			   (eq 'NO (car status)))
		  (error "Can't set ACL: %s" (cadr status))))))))))))

;;; Flag stuff.

(defun nnimap-save-info-hook ()
  (make-variable-buffer-local 'nnimap-saved-info)
  (setq nnimap-saved-info (copy-list (gnus-get-info gnus-newsgroup-name))))

(defun nnimap-update-flags-hook (&rest foo)
  (when (eq 'nnimap (car (gnus-find-method-for-group gnus-newsgroup-name)))
    (with-current-buffer gnus-summary-buffer
      (let ((group (gnus-group-real-name gnus-newsgroup-name))
	    (old-info nnimap-saved-info)
	    (new-info (gnus-get-info gnus-newsgroup-name))
	    delta-marks)

	;; Update read marks.
	
	(let* ((new-read (gnus-info-read new-info))
	       (old-read (gnus-info-read old-info))
	       (add (gnus-remove-from-range new-read (gnus-uncompress-range
						      old-read)))
	       (del (gnus-remove-from-range old-read (gnus-uncompress-range
						      new-read))))
	  (if add
	      (push (list add 'add '(read)) delta-marks))
	  (if del
	      (push (list del 'del '(read)) delta-marks)))
	
	;; Update marks.
	
	(let ((types gnus-article-mark-lists) 
	      type old-mark new-mark add del)
	  (while (setq type (cdr (pop types)))
	    ;; cache, score, bookmark are not proper flags.
	    (unless (memq type '(cache score bookmark))
	      (setq old-mark (cdr (assq type (gnus-info-marks old-info)))
		    new-mark (cdr (assq type (gnus-info-marks new-info)))
		    add (gnus-remove-from-range new-mark (gnus-uncompress-range
							  old-mark))
		    del (gnus-remove-from-range old-mark (gnus-uncompress-range
							  new-mark)))
	      (if add
		  (push (list add 'add (list type)) delta-marks))
	      (if del
		  (push (list del 'del (list type)) delta-marks)))))
	
	(nnimap-request-set-mark group delta-marks)
	(nnimap-expunge-close-group)))))

;;; Gnus glue

(defun nnimap-group-mode-hook ()
  (define-key gnus-group-mode-map (if (fboundp 'kbd) (kbd "G l")
				    (read-kbd-macro "G l"))
    'gnus-group-nnimap-edit-acl)
  (define-key gnus-group-mode-map (if (fboundp 'kbd) (kbd "G x")
				    (read-kbd-macro "G x"))
    'gnus-group-nnimap-expunge))
(add-hook 'gnus-group-mode-hook 'nnimap-group-mode-hook)

(add-hook 'gnus-summary-prepare-hook 'nnimap-save-info-hook)
(add-hook 'gnus-summary-exit-hook 'nnimap-update-flags-hook)

;; We're done.

(provide 'nnimap)
